(ns cerber.gatekeeper
  (:require [cerber.db.schema.recipies :refer [hashid-opts]]
            [cerber.db.store :as store]
            [cerber.http.response :as response]
            [clauth
             [client :as client]
             [endpoints :as endpoints]
             [user :as user]]
            [clojure.tools.logging :as log]
            [compojure.core :refer [ANY GET POST routes]]))

(defn create-client
  "Creates default client basing on its name, url and id"

  [{:keys [name url client-id client-secret]}]
  (let [client (client/client-app name url)]
    (assoc client
           :client-id client-id
           :client-secret client-secret)))

(defn authenticate-user
  "Checks whether user :is_active and tries to authenticates with provided login and password.
  Returns user's data in case of successful authentication or nil otherwise."

  [login password]
  (when-let [user (store/fetch-user-by-login login)]
    (when-let [p (:password user)]
      (if (and (:is_active user)
               (user/valid-password? password p)) user))))

(defn authorization-form
  "Renders authorization form asking for permission to access user's resources"

  ([req]
   (let [{{:keys [client_id response_type redirect_uri scope state]} :params} req
         client (client/fetch-client client_id)]

     (response/render-form "oauth/authorize.html" {:client (:name client)
                                                   :url (req :uri)
                                                   :client-id client_id
                                                   :response-type response_type
                                                   :redirect-uri redirect_uri
                                                   :scope scope
                                                   :state state} req))))

(defn login-form
  "Renders login form"

  ([req]
   (login-form req (req :uri) ((req :params) :login) ((req :params) :password)))
  ([req uri login password]
   (response/render-form "oauth/login.html" {:login login
                                             :pass  password
                                             :url   (req :uri)} req)))

(defn init-client
  "Initializes default OAuth client"

  [client-config]
  (try
    (or (client/fetch-client (:client-id client-config))
        (client/store-client (create-client client-config)))
    (catch java.net.ConnectException e
      (log/error "You don't have stores configured (init-stores?) or a database running in the background:" (.getMessage e)))))

(defn init-stores
  [stores-config]
  (store/create-ident-store stores-config)
  (store/create-token-store stores-config))

(defn init-routes
  "Initializes Gatekeeper routes."

  [client config]
  (when-let [hashids-config (:hashids config)]
    (reset! hashid-opts hashids-config))

  (routes
    (GET  "/logout"    [] endpoints/logout-handler)
    (ANY  "/authorize" [] (endpoints/authorization-handler {:authorization-form authorization-form}))
    (POST "/token"     [] (endpoints/token-handler {:user-authenticator authenticate-user}))
    (ANY  "/login"     [] (endpoints/login-handler {:user-authenticator authenticate-user
                                                    :login-destination (:landing-page config)
                                                    :login-form login-form
                                                    :client client}))))
