(ns cerber.authenticator
  (:require [compojure.core        :refer [POST GET routes]]
            [clojure.string        :as str]
            [clojure.tools.logging :as log]
            [cerber.db.store       :as store]
            [cerber.http.response  :as response]
            [cheshire.core         :as json]
            [clauth.user           :as user]
            [clauth.token          :as token]
            [clj-http.client       :as client]
            [clj-oauth2.client     :as oauth2]))

(defn normalize-login
  "Normalizing login to required rules:
  - all lower-cased,
  - starts with letter followed by letters, digits, dot or underscore"

  [login]
  (re-find #"[a-z][a-z0-9\_\.]+" (-> login
                                     (str/lower-case)
                                     (str/replace #"[!@#\$\%\^\&\*\(\)\+\-\?<>]+" "")
                                     (str/replace " " "."))))

(defn generate-token
  "Generates OAuth token for given OAuth client wired with user's identifier"

  [client user]
  (token/create-token client user))

(defn request-access-token [config auth-req]
  (try
    (if-let [response (oauth2/get-access-token config (:params auth-req) auth-req)]
      (:access-token response))
    (catch Exception e
      (log/error "Cannot receive access-token. " (.getMessage e)))))

(defn request-user-details [url headers access-token]
  "Returns user's data from oAuth provider (assuming valid access-token)"

  (json/parse-string
   (let [response (client/get (str/replace-first url "{token}" access-token) {:headers headers})]
     (:body response))))

(defn authentication-handler [config client req]
  (if-let [token  (request-access-token config req)]
    (let [aboutme (:about-me config)
          details (request-user-details (:url aboutme) (:headers aboutme) token)]

      ;; check if user with given oauth id already exsits
      ;; if so, create token and let him in. otherwise redirect to register form.

      (if-let [user (store/fetch-user-by-oid (details (:id aboutme)))]
        (response/redirect-to (:landing-page config) {:access_token (:token (generate-token client user))})
        (response/render-form "oauth/register.html"  {:token  token
                                                      :email  (details (:email aboutme))
                                                      :login  (normalize-login (details (:name aboutme)))
                                                      :action "/auth/register"
                                                      :url    "/auth/login"} req)))
    ;; something went wrong - let the user know.
    ;; usually it's enough just to log in again.

    (response/render-page "oauth/exception.html")))

(defn login-handler [config req]
  (response/redirect-to
   (:uri (oauth2/make-auth-request config))))

(defn registration-handler [{aboutme :about-me} {{:keys [token login email]} :params}]
  (when-let [details (request-user-details (:url aboutme) (:headers aboutme) token)]

    ;; register user with given OAuth id
    ;; if he doesn't exist yet.

    (if (store/fetch-user-by-login login)

      (response/user-exists)
      (user/register-user (store/bare-new-user {:login login
                                                :email email
                                                :oid (details (:id aboutme))
                                                :active? true})))))

(defn init-routes
  "Initializes Authenticator routes"

  [client config]
  (routes
   (GET  "/auth/callback" [] (partial authentication-handler config client))
   (GET  "/auth/login"    [] (partial login-handler config))
   (POST "/auth/register" [] (partial registration-handler config))))
