(ns matthiasn.systems-toolbox.component
                    
         (:require-macros [cljs.core.async.macros :refer [go-loop]])
  (:require
                                             
           [cljs.core.match :refer-macros [match]]
           [matthiasn.systems-toolbox.helpers :refer [request-animation-frame]]
                                                                                                                                   
           [cljs.core.async :refer [<! >! chan put! sub pipe mult tap pub buffer sliding-buffer dropping-buffer timeout]]
                                         ))

                                               
       (defn now [] (.getTime (js/Date.)))

(defn make-chan-w-buf
  "Create a channel with a buffer of the specified size and type."
  [config]
  (match config
         [:sliding  n]  (chan (sliding-buffer n))
         [:buffer   n]  (chan (buffer n))
         :else          (prn "invalid: " config)))

(def component-defaults
  {:in-chan  [:buffer 1]  :sliding-in-chan  [:sliding 1]  :throttle-ms 1
   :out-chan [:buffer 1]  :sliding-out-chan [:sliding 1]  :firehose-chan  [:buffer 1]})

(defn msg-handler-loop
  "Constructs a map with a channel for the provided channel keyword, with the buffer
  configured according to cfg for the channel keyword. Then starts loop for taking messages
  off the returned channel and calling the provided handler-fn with the msg.
  Does not process return values from the processing step; instead, put-fn needs to be
  called to produce output."
  [state handler-fn put-fn cfg cmp-id chan-key firehose-chan]
  (when handler-fn
    (let [chan (make-chan-w-buf (chan-key cfg))]
      (go-loop []
        (let [msg (<! chan)
              msg-meta (-> (merge (meta msg) {})
                           (assoc-in , [:cmp-seq] cmp-id) ; TODO: replace by actual sequence
                           (assoc-in , [cmp-id :in-timestamp] (now)))
              [msg-type _] msg]
          (when-not (= "firehose" (namespace msg-type))
            (put! firehose-chan [:firehose/cmp-recv {:cmp-id cmp-id :msg msg}]))
          (if (= msg-type :cmd/get-state)
            (put-fn [:state/snapshot {:cmp-id cmp-id :snapshot @state}])
            (handler-fn state put-fn (with-meta msg msg-meta)))
          (when (= chan-key :sliding-in-chan) (<! (timeout (:throttle-ms cfg))))
          (recur)))
      {chan-key chan})))

(defn make-component
  "Creates a component with attached in-chan, out-chan, sliding-in-chan and sliding-out-chan.
  It takes the initial state atom, the handler function for messages on in-chan, and the
  sliding-handler function, which handles messages on sliding-in-chan.
  By default, in-chan and out-chan have standard buffers of size one, whereas sliding-in-chan
  and sliding-out-chan have sliding buffers of size one.
  The buffer sizes can be configured.
  The sliding-channels are meant for events where only ever the latest version is of interest,
  such as mouse moves or published state snapshots in the case of UI components rendering
  state snapshots from other components."
  ([cmp-id mk-state handler sliding-handler]
   (make-component cmp-id mk-state handler sliding-handler component-defaults))

  ([cmp-id mk-state handler sliding-handler opts]
   (let [cfg (merge component-defaults opts)
         out-chan (make-chan-w-buf (:out-chan cfg))
         firehose-chan (make-chan-w-buf (:firehose-chan cfg))
         out-pub-chan (make-chan-w-buf (:out-chan cfg))
         sliding-out-chan (make-chan-w-buf (:sliding-out-chan cfg))
         put-fn (fn [msg]
                  (let [msg-meta (-> (merge (meta msg) {})
                                     (assoc-in , [:cmp-seq] cmp-id) ; TODO: replace by actual sequence
                                     (assoc-in , [cmp-id :out-timestamp] (now)))
                        msg-w-meta (with-meta msg msg-meta)]
                    (put! out-chan msg-w-meta)
                    (put! firehose-chan [:firehose/cmp-put {:cmp-id cmp-id :msg msg-w-meta}])))
         out-mult (mult out-chan)
         firehose-mult (mult firehose-chan)
         state (mk-state put-fn)
         watch-state (if-let [watch (:watch cfg)] (watch state) state)
         changed (atom true)]
     (tap out-mult out-pub-chan)

          
         
                             
                          
                                       
                                                                                               
                                                                                                  

           
     (letfn [(step []
                   (request-animation-frame step)
                   (when @changed
                     (put! sliding-out-chan (with-meta [:app-state @watch-state] {:from cmp-id}))
                     (swap! changed not)))]
       (request-animation-frame step)
       (try
         (add-watch watch-state
                    :watcher
                    (fn [_ _ _ new-state]
                      (reset! changed true)))
         (catch js/Object e (prn cmp-id " Exception attempting to watch atom: " watch-state e))))

     (merge
       {:out-mult out-mult
        :firehose-chan firehose-chan
        :firehose-mult firehose-mult
        :out-pub (pub out-pub-chan first)
        :state-pub (pub sliding-out-chan first)
        :cmp-id cmp-id
        :state-snapshot-fn (fn [] @watch-state)}
       (msg-handler-loop state handler put-fn cfg cmp-id :in-chan firehose-chan)
       (msg-handler-loop state sliding-handler put-fn cfg cmp-id :sliding-in-chan firehose-chan)))))

;;;;;;;;;;;; This file autogenerated from src/cljx/matthiasn/systems_toolbox/component.cljx
