(ns matteoredaelli.qliksense.core
  (:require
   [clojure.spec.alpha :as s]
   [expound.alpha :as expound]
   [clojure.tools.logging :as log]
   [cheshire.core :refer :all]
   [matteoredaelli.qliksense.api :as api]
   [matteoredaelli.qliksense.import :as import]
   [matteoredaelli.qliksense.utils :as utils]
   [matteoredaelli.qliksense.license :as cmd-license]
   [matteoredaelli.qliksense.user :as cmd-user]
   [matteoredaelli.qliksense.systemrule :as cmd-systemrule]
   [cli-matic.core :refer [run-cmd]])
  (:gen-class)
)

(s/check-asserts true)

(defn get-system-map [config system]
  ""
  (let [
        systems (s/assert map? (:systems config))
        system_map (s/assert map? (system systems))]
    system_map
    ))


(defn delete-request
  "http get request"
  [{:keys [config system url params outfile]}]
  (let [system_map (get-system-map config system)
        resp (api/delete-request system_map url params)
        body (resp :body)]
    (utils/out-string body outfile)))


(defn get-request
  "http get request"
  [{:keys [config system url params outfile]}]
  (let [system_map (get-system-map config system)
        resp (api/get-request system_map url params)
        body (resp :body)]
    (utils/out-string body outfile)))


(defn post-request
  "http get request"
  [{:keys [config system url params body outfile]}]
  (let [system_map (get-system-map config system)
        resp (api/post-request system_map url params body)
        body (resp :body)]

    (utils/out-string body outfile)))

(def opt-body  {:as "body"
                :option "body"
                :short "b"
                :default :present
                :type :jsonfile})

(def opt-ids {:as     "using existing IDs"
              :option "ids"
              :short  "I"
              :default false
              :type :with-flag})

(def opt-infile {:as "a json file name"
                   :option  "infile"
                   :short   "i"
                   :type    :jsonfile})
(def opt-outfile {:as "a json file name"
                  :option  "outfile"
                  :default nil
                  :short   "o"
                  :type    :string})

(def opt-resource-type {:as "Resource type"
                  :option  "resource-type"
                  :default nil
                  :short   "T"
                  :type    :string})

(def opt-resource-filter {:as "Resource filter"
                     :option  "resource-filter"
                     :default nil
                     :short   "F"
                     :type    :string})

(def opt-user-filter {:as "User filter"
                     :option  "user-filter"
                     :default nil
                     :short   "U"
                     :type    :string})

(def opt-url {:as     "url url for the http request"
                   :option "url"
                   :short  "p"
                   :default :present
                   :type    :string})

(def opt-params {:as     "params"
                     :option "params"
                     :type   :json})

(def opt-run {:as      "really run or dry run "
                  :option  "run"
                  :type    :with-flag
              :default false})

(def opt-system {:as      "system name defined inside the config file"
                     :option  "system"
                     :short   "s"
                     :default :present
                 :type    :keyword})

(def CONFIGURATION
  {:command     "qliksense"
   :description "A command-line tool for interacting with Qliksense repository APIs"
   :version     "0.3.0"
   :opts        [{:as      "a json config file"
                  :default "config.edn"
                  :option  "config"
                  :type    :ednfile}]

   :subcommands [
                 {:command     "delete-request"
                  :description "generic http DELETE request"
                  :opts        [opt-system
                                opt-url
                                opt-params
                                opt-outfile
                                ]
                  :runs        delete-request}
                 {:command     "get-request"
                  :description "generic http GET request"
                  :opts        [opt-system
                                opt-outfile
                                opt-url
                                opt-params]
                  :runs        get-request}
                 {:command     "import-many"
                  :description "(raw) import many entities"
                  :opts        [opt-system
                                opt-url
                                opt-infile
                                opt-ids
                                opt-run]
                  :runs        import/import-many}
                 {:command     "post-request"
                  :description "generic http POST request"
                  :opts        [opt-system
                                opt-url
                                opt-params
                                opt-body
                                opt-outfile
                                ]
                  :runs        post-request}
                 {:command     "rm-double-licenses"
                  :description "Deallocate analyzer licenses for professional users"
                  :opts        [opt-system
                                opt-url
                                opt-outfile
                                opt-run
                                ]
                  :runs        cmd-license/remove-analyzer-licenses-for-professional-users}
                 {:command     "rm-old-users"
                  :description "Delete externally removed users"
                  :opts        [opt-system
                                opt-url
                                opt-run
                                ]
                  :runs        cmd-user/rm-externally-removed-users}

                 {:command     "security-audit-matrix"
                  :description "Delete externally removed users"
                  :opts        [opt-system
                                opt-url
                                opt-resource-type
                                opt-resource-filter
                                opt-user-filter
                                opt-outfile
                                ]
                  :runs        cmd-systemrule/systemrule-security-audit-matrix}
                 ]})

(defn -main
  "I don't do a whole lot ... yet."
  [& args]
  (binding [*out* *err*]
    (println  """ ___  _ _ _                               _
 / _ \\| (_) | _____  ___ _ __  ___  ___   (_) __ _ _ __
| | | | | | |/ / __|/ _ \\ '_ \\/ __|/ _ \\  | |/ _` | '__|
| |_| | | |   <\\__ \\  __/ | | \\__ \\  __/_ | | (_| | |
 \\__\\_\\_|_|_|\\_\\___/\\___|_| |_|___/\\___(_)/ |\\__,_|_|
                                        |__/

"""))
  (run-cmd args CONFIGURATION)
  )
