(ns qliksense.core
  (:require
   [clojure.spec.alpha :as s]
   [expound.alpha :as expound]
   [clojure.tools.logging :as log]
   [cheshire.core :refer :all]
   [qliksense.api :as api]
   [qliksense.import :as import]
   [qliksense.utils :as utils]
   [qliksense.license :as cmd-license]
   [qliksense.user :as cmd-user]
   [cli-matic.core :refer [run-cmd]])
  (:gen-class)
)

(s/check-asserts true)

(defn get-system-map [config system]
  ""
  (let [
        systems (s/assert map? (config "systems"))
        system_map (s/assert map? (systems system))]
    system_map
    ))


(defn about
  "s"
  [{:keys [config system outfile]}]
  (let [
        system_map (get-system-map config system)
        resp (api/get-request system_map "/qrs/about" {})
        ]
    (utils/out-string resp outfile)))

(defn delete-request
  "http get request"
  [{:keys [config system path params outfile]}]
  (let [system_map (get-system-map config system)
        resp (api/delete-request system_map path params)
        body (resp :body)]
    (utils/out-string body outfile)))


(defn get-request
  "http get request"
  [{:keys [config system path params outfile]}]
  (let [system_map (get-system-map config system)
        resp (api/get-request system_map path params)
        body (resp :body)]
    (utils/out-string body outfile)))


(defn post-request
  "http get request"
  [{:keys [config system path params body outfile]}]
  (let [system_map (get-system-map config system)
        resp (api/post-request system_map path params body)
        body (resp :body)]

    (utils/out-string body outfile)))

(def opt-body  {:as "body"
                :option "body"
                :short "b"
                :default :present
                :type :jsonfile})

(def opt-ids {:as     "using existing IDs"
              :option "ids"
              :short  "I"
              :default false
              :type :with-flag})

(def opt-infile {:as "a json file name"
                   :option  "infile"
                   :short   "i"
                   :type    :jsonfile})

(def opt-outfile {:as "a json file name"
                  :option  "outfile"
                  :default nil
                  :short   "o"
                  :type    :string})

(def opt-path {:as     "url path for the http request"
                   :option "path"
                   :short  "p"
                   :default :present
                   :type    :string})

(def opt-params {:as     "params"
                     :option "params"
                     :type   :json})

(def opt-run {:as      "really run or dry run "
                  :option  "run"
                  :type    :with-flag
              :default false})

(def opt-system {:as      "system name defined inside the config file"
                     :option  "system"
                     :short   "s"
                     :default :present
                 :type    :string})

(def CONFIGURATION
  {:command     "qliksense"
   :description "A command-line tool for interacting with Qliksense repository APIs"
   :version     "0.2.0"
   :opts        [{:as      "a json config file"
                  :default "config.json"
                  :option  "config"
                  :type    :jsonfile}]

   :subcommands [{:command     "about"
                  :description "About"
                  :opts        [opt-system
                                opt-outfile
                                ]
                  :runs        about}
                 {:command     "delete-request"
                  :description "generic http DELETE request"
                  :opts        [opt-system
                                opt-path
                                opt-params
                                opt-outfile
                                ]
                  :runs        delete-request}
                 {:command     "get-request"
                  :description "generic http GET request"
                  :opts        [opt-system
                                opt-outfile
                                opt-path
                                opt-params]
                  :runs        get-request}
                 {:command     "import-many"
                  :description "(raw) import many entities"
                  :opts        [opt-system
                                opt-infile
                                opt-ids
                                opt-run]
                  :runs        import/import-many}
                 {:command     "post-request"
                  :description "generic http POST request"
                  :opts        [opt-system
                                opt-path
                                opt-params
                                opt-body
                                opt-outfile
                                ]
                  :runs        post-request}
                 {:command     "rm-double-licenses"
                  :description "Deallocate analyzer licenses for professional users"
                  :opts        [opt-system
                                opt-outfile
                                opt-run
                                ]
                  :runs        cmd-license/remove-analyzer-licenses-for-professional-users}
                 {:command     "rm-old-users"
                  :description "Delete externally removed users"
                  :opts        [opt-system
                                opt-run
                                ]
                  :runs        cmd-user/rm-externally-removed-users}

                 ]})

(defn -main
  "I don't do a whole lot ... yet."
  [& args]
  (binding [*out* *err*]
    (println  """ ___  _ _ _                               _
 / _ \\| (_) | _____  ___ _ __  ___  ___   (_) __ _ _ __
| | | | | | |/ / __|/ _ \\ '_ \\/ __|/ _ \\  | |/ _` | '__|
| |_| | | |   <\\__ \\  __/ | | \\__ \\  __/_ | | (_| | |
 \\__\\_\\_|_|_|\\_\\___/\\___|_| |_|___/\\___(_)/ |\\__,_|_|
                                        |__/

"""))
  (run-cmd args CONFIGURATION)
  )
