(ns materia.db
  (:require [clojure.string :as str]
            [inflections.core :as inf]
            [jdbc.core :as jdbc]
            [materia.logging :as log]
            [materia.sql :as sql]
            [materia.util :as u]))

(def ^:dynamic *connection-source*)
(def ^:dynamic *connection*)
(def ^:dynamic *print-sql?* false)

(defmacro with-connection-source
  "Binds a connection source to `*connection-source*`. The connection
  source can be any types `jdbc.core/connection` can accept."
  [connection-source & body]
  `(binding [*connection-source* ~connection-source]
     ~@body))

(defn connection
  "Creates a connection to a database from `*connection-source*`. You
  need to bind a valid value to `*connection-source*` before using
  this."
  ([]
   (connection {}))
  ([opts]
   (assert
    (and (bound? #'*connection-source*)
         *connection-source*)
    "*connection-source* must be non-nil.")
   (jdbc/connection *connection-source* opts)))

(defmacro with-connection
  "Creates and opens a new connection to a database and binds it to
  `*connection*`."
  [& body]
  `(with-open [conn# (connection)]
     (binding [*connection* conn#]
       ~@body)))

(defmacro with-connection-dwim
  "Same as `with-connection`, but reuses a connection if
  possible."
  [& body]
  `(do
     (assert
      (some (every-pred bound? deref) [#'*connection* #'*connection-source*])
      "At least one of *connection-source* or *connection* must be non-nil.")
     (if (bound? #'*connection*)
       (do ~@body)
       (with-connection ~@body))))

(defmacro atomic
  "Evaluates body in transaction."
  [& body]
  (if (map? (first body))
    `(with-connection-dwim
       (let [conn# *connection*]
         (jdbc/atomic conn# ~(first body)
                      (binding [*connection* conn#]
                        ~@(next body)))))
    `(with-connection-dwim
       (let [conn# *connection*]
         (jdbc/atomic conn#
                      (binding [*connection* conn#]
                        ~@body))))))

(defn set-rollback!
  "Mark a current connection for rollback."
  []
  (jdbc/set-rollback! *connection*))

(defn- print-sql [sql]
  (log/info (pr-str sql)))

(defn- print-sql-dwim [sql]
  (when *print-sql?*
    (print-sql sql)))

(defmacro with-sql-logger
  "Evaluates body with an SQL logger that prints all SQLs executed via
  `execute` and `fetch`."
  [& body]
  `(binding [*print-sql?* true]
     ~@body))

(defn execute
  "Executes a query and returns a number of rows affected or inserted
  keys. It can accept a raw sql, a sqlvec format, or a map created via
  `stch.sql` DSL."
  ([q]
   (execute q {:returning :all}))
  ([q opt]
   (let [sql (sql/sql-format q {})]
     (print-sql-dwim sql)
     (with-connection-dwim
       (jdbc/execute-prepared! *connection* sql opt)))))

(defn fetch
  "Fetches results executing a query. It can accept a raw sql, a
  sqlvec format, or a map created via `stch.sql` DSL."
  ([q]
   (fetch q {:identifiers (comp str/lower-case inf/dasherize)}))
  ([q opt]
   (let [sql (sql/sql-format q {:quoting :mysql})] ; FIXME: Make quoting style configurable
     (print-sql-dwim sql)
     (with-connection-dwim
       (jdbc/fetch *connection* sql opt)))))

(u/with-arglists #'fetch
  (def fetch-one
  "Fetches one result executing a query. See also `fetch`."
  (comp first fetch)))

(u/with-arglists #'defn
  (defmacro deffetcher [query-name & rest-fn-spec]
    (let [query-name* (symbol (str (name query-name) "*"))]
      `(do (defn ~query-name* ~@rest-fn-spec)
           (u/with-arglists (var ~query-name*)
             (def ~query-name (comp fetch ~query-name*)))))))

(u/with-arglists #'defn
  (defmacro defexecutor [query-name & rest-fn-spec]
    (let [query-name* (symbol (str (name query-name) "*"))]
      `(do (defn ~query-name* ~@rest-fn-spec)
           (u/with-arglists (var ~query-name*)
             (def ~query-name (comp execute ~query-name*)))))))

(defn current-db-name
  "Returns database name of current source."
  []
  (get-in *connection-source* [:dbspec :name]))

(defn current-vendor
  "Returns vendor name of current source."
  []
  (get-in *connection-source* [:dbspec :vendor]))

(defexecutor drop-table!
  "Drop a table."
  [table-name]
  (format "drop table if exists `%s`" table-name))

(defn- ensure-mysql []
  (assert (= (current-vendor) "mysql")))

(defn get-tables
  "Returns all tables belonging to specified database. (Currently only
  works with MySQL)"
  [db-name]
  (ensure-mysql)
  (-> (sql/select :table-name)
      (sql/from :information-schema.tables)
      (sql/where `(= table-schema ~db-name))
      fetch
      (->> (map :table-name))))

(defn drop-all-tables!
  "Drop all tables of current source. (Currently only works with
  MySQL)"
  []
  (ensure-mysql)
  (let [qs (map drop-table!* (get-tables (current-db-name)))]
    (with-connection-dwim
      (execute "SET FOREIGN_KEY_CHECKS = 0")
      (doseq [q qs]
        (execute q))
      (execute "SET FOREIGN_KEY_CHECKS = 1"))))
