(ns materia.store
  (:refer-clojure :exclude [find count])
  (:require [clojure.core :as core]
            [clojure.string :as str]
            [inflections.core :as inf]
            [materia.db :as db]
            [materia.sql :as sql]))

;;;
;;; Extensible meta helpers
;;;

(defn- get-class-dwim [class-or-object]
  (let [klass (type class-or-object)]
    (if (= klass java.lang.Class)
      class-or-object
      klass)))

(defmulti pk get-class-dwim)

(defmethod pk :default [class-or-object]
  :id)

(defmulti table get-class-dwim)

(defmethod table :default [class-or-object]
  (-> class-or-object
      get-class-dwim
      .getName
      (str/split #"\.")
      last
      inf/underscore
      keyword))

(defmethod table clojure.lang.Keyword [k] k)

(defmethod table java.lang.String [s] (keyword s))

(defn pk-val [entity]
  (get entity (pk entity)))

;;;
;;; Hooks
;;;

(defmulti pre-find  get-class-dwim)
(defmulti post-find get-class-dwim)

(defmulti pre-save  get-class-dwim)
(defmulti post-save get-class-dwim)

(defmulti pre-delete  get-class-dwim)
(defmulti post-delete get-class-dwim)


;; default impls

(defmethod pre-find  :default [_] identity)
(defmethod post-find :default [_] identity)

(defmethod pre-save  :default [_] identity)
(defmethod post-save :default [_] identity)

(defmethod pre-delete  :default [_] identity)
(defmethod post-delete :default [_] identity)


;; Helpers to write hooks

(defn wrap-pre-save-modifier
  "A helper to write a pre-save hook"
  [modifier]
  (fn [dml]
    (cond
      (contains? dml :values)
      (update-in dml [:values] (partial map modifier))

      (contains? dml :set)
      (update-in dml [:set] modifier)

      :else dml)))


;;;
;;; DML helpers
;;;

(defn update* [entity]
  (let [id (pk-val entity)]
    (assert id)
    (-> (sql/update (table entity))
        (sql/setv entity)
        (sql/where (list '= (symbol (name (pk entity))) id)))))

(defn create* [entity]
  (-> (sql/insert-into (table entity))
      (sql/values [entity])))

(defn save* [entity]
  (if (pk-val entity)
    (update* entity)
    (create* entity)))

(defn delete* [entity]
  (let [id (pk-val entity)]
    (assert id)
    (-> (sql/delete-from (table entity))
        (sql/where (list '= (symbol (name (pk entity))) id)))))

(defn save [entity]
  (-> entity
      save*
      ((pre-save entity))
      db/execute
      ((post-save entity))
      first
      vals
      first))

(defn delete [entity]
  (-> entity
      delete*
      ((pre-delete entity))
      db/execute
      ((post-delete entity))
      first))


;;;
;;; Finder
;;;

(defn expand-query [q]
  (cond
    (nil? q)
    [:$= [1 1]]

    (map? q)
    (expand-query (map expand-query q))

    (sequential? q)
    (let [[k v] q]
      (cond
        (not (keyword? k))
        (if (= 1 (core/count q))
          (expand-query (first q))
          [:$and (map expand-query q)])

        (= k :$and)
        (expand-query v)

        (= k :$or)
        (cons :$or (rest (expand-query v)))

        (= k :$not)
        [:$not (expand-query v)]

        (re-find #"^\$" (name k))
        q

        ;; syntax sugars

        (and (string? v) (re-find #"%" v))
        [:$like [k v]]

        (instance? java.util.regex.Pattern v)
        [:$regex [k (str v)]]

        (vector? v)
        [:$in [k v]]

        :else
        [:$= q]))

    :else
    (throw (java.lang.IllegalArgumentException. (format "Cannot expand a query - %s" q)))))

(defmulti handle-query-op first)

(defmulti handle-option (fn [q [k v]] k))

(defn handle-options [q opts]
  (reduce handle-option q opts))

(defn find* [klass & [query-map opts]]
  (-> (sql/select :*)
      (sql/from (table klass))
      (sql/where (handle-query-op (vec (expand-query query-map))))
      (handle-options opts)
      ((pre-find klass))))

(defn find [klass & [query-map opts]]
  (-> (find* klass query-map opts)
      db/fetch
      ((post-find klass))))

(defn find-first [klass & [query-map opts]]
  (first (find klass query-map (assoc opts :limit 1))))

(defn count [klass & [query-map opts]]
  (-> (find* klass query-map opts)
      (sql/replace-select '(count :*))
      db/fetch
      first
      vals
      first))

;; query ops

(defmethod handle-query-op :$and [[op v]]
  `(~'and ~@(map handle-query-op v)))

(defmethod handle-query-op :$or [[op v]]
  `(~'or ~@(map handle-query-op v)))

(defmethod handle-query-op :$not [[op v]]
  `(~'not ~(handle-query-op v)))

(defmacro op-mapper [ops]
  `(do
     ~@(for [op ops]
         `(defmethod handle-query-op ~(keyword (str "$" (name op))) [[~'op [~'k & ~'args]]]
            (apply list (symbol ~(name op)) (if (keyword? ~'k) (symbol (name ~'k)) ~'k) ~'args)))))

(op-mapper [:= :!= :not= :is :is-not :in :not-in :like :not-like :regex :regexp :<>
            :< :<= :> :>= :between :not-between])

;; query options
;;  TODO: add more options

(defmethod handle-option :limit [q [_ v]]
  (sql/limit q v))

(defmethod handle-option :offset [q [_ v]]
  (sql/offset q v))

(defmethod handle-option :order-by [q [_ v]]
  (if (and (sequential? v)
           (not (#{:asc :desc} (last v))))
    (apply sql/order-by q v)
    (sql/order-by q v)))

(defmethod handle-option :selects [q [_ vs]]
  (reduce sql/select (sql/replace-select q (first vs)) (rest vs)))

(defn generate-join [q {:keys [table alias condition] :as spec}]
  (-> q
      (sql/join (if alias [table alias] table) condition)))

(defmethod handle-option :joins [q [_ vs]]
  (reduce generate-join q vs))
