(ns materia.migration
  (:refer-clojure :exclude [int bigint boolean time update])
  (:require [clj-liquibase.change :refer :all]
            [clj-liquibase.cli :as cli]
            [clj-liquibase.core :refer [make-changelog]]
            [clj-time.core :refer [now default-time-zone]]
            [clj-time.format :refer [formatter unparse]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [inflections.core :as inf]
            [materia.util :as u]))

(defonce last-applied (atom nil))

;; TODO: Support more commands
(defn update [{:keys [datasource changelog chs-count
                      contexts sql-only]
               :as opts} & args]
  (apply cli/entry "update" opts args)
  (reset! last-applied changelog))

(defn rollback [{:keys [datasource changelog chs-count
                        tag date contexts sql-only]
                 :as opts} & args]
  (apply cli/entry "rollback" opts args))

;; for repl
(defn reupdate [{:keys [datasource changelog]
                 :as opts} & args]
  (when @last-applied
    (apply rollback {:datasource datasource :changelog @last-applied :chs-count "1"} args)
    (reset! last-applied nil))
  (apply update opts args))

(defn tag [{:keys [datasource tag]
            :as opts} & args]
  (apply cli/entry "tag" opts args))


;;; Helpers
;;   TODO: implement more helpers (at least, should cover basic column types)

(def default-column-config {:null false})

(defn column
  ([name type]
   (column name type {}))
  ([name type opt]
   `[~name ~type ~@(flatten (vec (merge default-column-config opt)))]))

(defn int [name & args]
  (apply column name :int args))

(defn bigint [name & args]
  (apply column name :bigint args))

(defn tinyint [name & args]
  (apply column name :tinyint args))

(defn boolean [name & args]
  (apply column name [:tinyint 1] args))

(defn id
  ([]
   (id false))
  ([big?]
   ((if big? bigint int)
    :id
    {:null    false
     :pk      true
     :autoinc true})))

(defn varchar [name & [{:keys [len]
                        :or {len 255}
                        :as opt}]]
  (column name [:varchar len] (dissoc opt :len)))

(defn text [name & args]
  (apply column name :text args))

(defn datetime [name & args]
  (apply column name :datetime args))

(defn time [name & args]
  (apply column name :time args))

(defn created-at []
  (datetime :created_at))

(defn updated-at []
  (datetime :updated_at))

(defn deleted-at []
  (datetime :deleted_at {:null true}))

(defn- generate-index-name [prefix ingredients]
  (str
   (str/upper-case prefix)
   "_"
   (str/join (map u/crc32-hex ingredients))))

(defn belongs-to [e to]
  (let [col (keyword (str (name to) "_id"))]
    [(add-columns e [(int col {:null true})])
     (create-index e [col] :unique false :index-name (generate-index-name "IDX" [e col]))
     (add-foreign-key-constraint (generate-index-name "FK" [e col]) e [col] to [:id])]))

(defonce migrations (atom []))

(defmacro defmigration [name & body]
  `(def ~(vary-meta name assoc :migration true)
     (swap! migrations conj [~(str name) "anon" (apply concat (map #(if (vector? %) % [%]) ~(vec body)))])))

(let [n *ns*]
  (defn collect-migrations [path]
    (reset! migrations [])
    (binding [*ns* n]
      (doseq [f (sort (filter #(re-find #"\.clj$" (.getName %)) (.listFiles (io/file path))))]
        (eval (read-string (str "(do " (slurp f) ")")))))
    @migrations))

(defn build-changelog [path]
  (when-let [log (seq (collect-migrations path))]
    (partial make-changelog "app" log)))

(defn- generate-empty-file-name [description]
  (let [fmt (formatter "yyyyMMddHHmmss" (default-time-zone))
        v   (unparse fmt (now))]
    (str "v_"
         v
         "_"
         (inf/underscore description)
         ".clj")))

(defn generate-empty-file [path description]
  (let [description (str/replace description #"\W" "-")
        f (io/file (str path "/"
                        (generate-empty-file-name description)))]
    (io/make-parents f)
    (spit f `(~'defmigration ~(symbol (inf/dasherize description))))
    f))
