(ns materia.repl
  (:require [clojure.java.io :as io]
            [clojure.tools.namespace.repl :refer [refresh]]
            [environ.core :refer [env]]
            [materia.config :refer [read-config]]
            [materia.db :as db]
            [materia.preset.system :refer [new-migration-system]]
            [materia.system :as sys]
            [materia.util.component :refer [with-component]])
  (:import java.nio.charset.Charset))

(defonce system nil)

(def config-file-path "config/config.edn")

(defn bundle-out! []
  (alter-var-root #'*out* (constantly *out*))
  (alter-var-root #'*err* (constantly *err*)))

(defn bundle-java-out! []
  (try
    (Class/forName "org.apache.commons.io.output.WriterOutputStream")
    (eval
     `(do
        (when (env :dev)
          (-> *out*
              (org.apache.commons.io.output.WriterOutputStream. (Charset/defaultCharset) 1024 true)
              java.io.PrintStream.
              System/setOut)
          (-> *err*
              (org.apache.commons.io.output.WriterOutputStream. (Charset/defaultCharset) 1024 true)
              java.io.PrintStream.
              System/setErr)
          )))
    (catch Exception e nil)))

(defn go []
  (bundle-out!)
  (sys/boot (read-config (io/resource config-file-path)) #'system)
  (alter-var-root #'db/*connection-source* (constantly (:db system)))
  system)

(defn stop []
  (sys/shutdown #'system)
  (alter-var-root #'db/*connection-source* (constantly nil))
  system)

(defn reset []
  (stop)
  (refresh :after 'materia.repl/go))

(defn reset* []
  (stop)
  (go))


;;; Migration tools

(defmacro with-migration-system [migrator & body]
  `(with-component [s# (new-migration-system (read-config (io/resource config-file-path)))]
     (let [~migrator (:migrator s#)]
       ~@body)))

(defn migrate []
  (with-migration-system m
    (.migrate m)))

(defn rollback [& [opts]]
  (with-migration-system m
    (.rollback m (or opts {:chs-count "1"}))))

(defn remigrate [& [opts]]
  (with-migration-system m
    (.remigrate m)))

(defn generate-migration-file [description]
  (with-migration-system m
    (.generate m description)))
