(ns materia.component.migrator
  (:require [materia.logging :as log]
            [materia.migration :as core]))

(defprotocol IMigrator
  (migrate [this])
  (remigrate [this])
  (rollback [this opts])
  (tag [this tag-name])
  (generate [this description]))

(defn- get-changelog [path]
  (if path (core/build-changelog path)))

(defrecord Migrator [conf db]
  IMigrator
  (migrate [this]
    (if-let [chlog (get-changelog (:path conf))]
      (do (core/update {:datasource (:datasource db)
                        :changelog chlog})
          ;; FIXME: Always print success message regardless of
          ;; whether new migration files exist or not.
          (log/info "Successfully migrated"))
      (log/info "No migrations to apply")))
  (remigrate [this]
    (if-let [chlog (get-changelog (:path conf))]
      (do (core/reupdate {:datasource (:datasource db)
                          :changelog chlog})
          (log/info "Successfully remigrated"))
      (log/info "No migrations to apply")))
  (rollback [this opts]
    (if-let [chlog (get-changelog (:path conf))]
      (do (core/rollback (merge {:datasource (:datasource db)
                                 :changelog  chlog}
                                opts))
          (log/info "Successfully rolled back"))
      (log/info "No migrations to roll back")))
  (tag [this tag-name]
    (core/tag {:datasource (:datasource db)
               :tag tag-name}))
  (generate [this description]
    (when-let [dir (:path conf)]
      (core/generate-empty-file dir description))))

(defn new-migrator [conf]
  (map->Migrator {:conf (:migration conf)}))
