(ns materia.repl
  (:require [clojure.java.io :as io]
            [clojure.tools.namespace.repl :refer [refresh]]
            [materia.config :refer [read-config]]
            [materia.services.app.service :refer :all]
            [materia.services.db.service :refer :all]
            [materia.services.db.core :as dcore]
            [materia.services.webserver.service :refer :all]
            [puppetlabs.trapperkeeper.app :as tka]
            [puppetlabs.trapperkeeper.bootstrap :as tkb]
            [puppetlabs.trapperkeeper.core :as tk]))

;; a var to hold the main `TrapperkeeperApp` instance.
(defonce system nil)

(def config-file-path "config/config.edn")

(defn init []
  (alter-var-root #'system
                  (fn [_] (tk/build-app
                           (tkb/parse-bootstrap-config! (tkb/find-bootstrap-config {}))
                           (read-config (io/resource config-file-path)))))

  (alter-var-root #'system tka/init)
  (tka/check-for-errors! system))

(defn start []
  (alter-var-root #'system
                  (fn [s] (if s (tka/start s))))
  (tka/check-for-errors! system))

(defn stop []
  (alter-var-root #'system
                  (fn [s] (when s (tka/stop s)))))

(defn go []
  (init)
  (start))

(defn get-system []
  (or system tk/main-app))

(defn context []
  @(tka/app-context (get-system)))

;; pretty print the entire application context
(defn print-context []
  (clojure.pprint/pprint (context)))

(defn service [service-id]
  (tka/get-service (get-system) service-id))

(defn reset []
  (stop)
  (refresh :after 'materia.repl/go))

(defmacro with-app
  [app services config & body]
  `(let [~app (tk/boot-services-with-config ~services ~config)]
     (try
       ~@body
       (finally
         (tka/stop ~app)))))

(defmacro with-app*
  [app services-file-path config-file-path & body]
  `(with-app ~app
     (tkb/parse-bootstrap-config! (io/resource ~services-file-path))
     (read-config (io/resource ~config-file-path))
     ~@body))

(defmacro with-db-service [& body]
  `(with-app ~'_
     [db-service]
     (read-config (io/resource config-file-path))
     (dcore/with-db
       ~@body)))

;;; Migration tools

(defn migrate []
  (with-app* app "migration.cfg" config-file-path
    (.migrate (tka/get-service app :MigrationService))))

(defn rollback [& [opts]]
  (with-app* app "migration.cfg" config-file-path
    (.rollback (tka/get-service app :MigrationService) (or opts {:chs-count "1"}))))

(defn remigrate [& [opts]]
  (with-app* app "migration.cfg" config-file-path
    (.remigrate (tka/get-service app :MigrationService))))

(defn generate-migration-file [description]
  (with-app* app "migration.cfg" config-file-path
    (.generate (tka/get-service app :MigrationService) description)))
