(ns materia.repl
  (:require [clojure.java.io :as io]
            [clojure.tools.namespace.repl :refer [refresh]]
            [materia.config :refer [read-config]]
            [materia.services.app.service :refer :all]
            [materia.services.db.service :refer :all]
            [materia.services.webserver.service :refer :all]
            [puppetlabs.trapperkeeper.app :as tka]
            [puppetlabs.trapperkeeper.bootstrap :as tkb]
            [puppetlabs.trapperkeeper.core :as tk]))

;; a var to hold the main `TrapperkeeperApp` instance.
(defonce system nil)

(defn init []
  (alter-var-root #'system
                  (fn [_] (tk/build-app
                           (tkb/parse-bootstrap-config! (tkb/find-bootstrap-config {}))
                           (read-config (io/resource "config/config.edn")))))

  (alter-var-root #'system tka/init)
  (tka/check-for-errors! system))

(defn start []
  (alter-var-root #'system
                  (fn [s] (if s (tka/start s))))
  (tka/check-for-errors! system))

(defn stop []
  (alter-var-root #'system
                  (fn [s] (when s (tka/stop s)))))

(defn go []
  (init)
  (start))

(defn get-system []
  (or system tk/main-app))

(defn context []
  @(tka/app-context (get-system)))

;; pretty print the entire application context
(defn print-context []
  (clojure.pprint/pprint (context)))

(defn service [service-id]
  (tka/get-service (get-system) service-id))

(defn reset []
  (stop)
  (refresh :after 'materia.repl/go))


;;; Migration tools

(defn migrate []
  (.migrate (service :MigrationService)))

(defn rollback [& [opts]]
  (.rollback (service :MigrationService) (or opts {:chs-count "1"})))

(defn remigrate [& [opts]]
  (.remigrate (service :MigrationService)))

(defn generate-migration-file [description]
  (.generate (service :MigrationService) description))
