(ns materia.middleware
  (:require [clojure.string :as str]
            [materia.cljs.injector :refer [wrap-js-injector]]
            [materia.request :refer [wrap-request-binding]]
            [materia.services.db.core :refer [wrap-sql-logger]]
            [materia.utils :as u]
            [prone.middleware :refer [wrap-exceptions]]
            [ring.middleware.defaults :refer :all]
            [ring.middleware.reload :refer [wrap-reload]]
            [taoensso.timbre :as log]))

(defn wrap-request-logger [handler]
  (fn [req]
    (log/info (with-out-str (clojure.pprint/pprint req)))
    (handler req)))

(defn default-middleware [conf]
  #(wrap-defaults % conf))

(defn- contain-meta? [ns ks]
  (some #(get-in (meta %) ks) (vals (ns-interns ns))))

(defn- contain-layout? [ns]
  (contain-meta? ns [:layout]))

(defn- contain-snippet? [ns]
  (contain-meta? ns [:snippet]))

(defn- find-ns-contains-templates []
  (->> (all-ns)
       (filter (some-fn contain-layout? contain-snippet?))
       (map ns-name)))

(defn wrap-template-reload [handler]
  (fn [req]
    (doseq [ns (find-ns-contains-templates)]
      (require ns :reload))
    (handler req)))

(defn wrap-ignore-trailing-slash
  "Modifies the request uri before calling the handler.
  Removes a single trailing slash from the end of the uri if present.

  Useful for handling optional trailing slashes until Compojure's route matching syntax supports regex.
  Adapted from http://stackoverflow.com/questions/8380468/compojure-regex-for-matching-a-trailing-slash"
  [handler]
  (fn [{uri :uri :as request}]
    (handler (assoc request :uri (cond (= "/" uri) uri
                                       (.endsWith uri "/") (apply str (butlast uri))
                                       :else uri)))))

(defn middlewares [{:keys [request-logger
                           request-binding
                           auto-reload
                           prone sql-logger
                           inject-js]
                    :as conf}]
  (u/conj-> []
            (and inject-js
                 (seq inject-js))
            [20 #(wrap-js-injector % inject-js)]

            request-logger
            [30 wrap-request-logger]

            request-binding
            [30 wrap-request-binding]

            auto-reload
            [100 wrap-reload]

            auto-reload
            [100 wrap-template-reload]

            prone
            [200 wrap-exceptions]

            sql-logger
            [100 wrap-sql-logger]

            true
            [100 (default-middleware conf)]

            true
            [200 wrap-ignore-trailing-slash]))
