(ns materia.store
  (:require [clojure.string :as str]
            [inflections.core :as inf]
            [materia.services.db.core :as m]
            [stch.sql :refer :all]))

;;;
;;; Extensible meta helpers
;;;

(defn- get-class-dwim [class-or-object]
  (let [klass (type class-or-object)]
    (if (= klass java.lang.Class)
      class-or-object
      klass)))

(defmulti pk get-class-dwim)

(defmethod pk :default [class-or-object]
  :id)

(defmulti table get-class-dwim)

(defmethod table :default [class-or-object]
  (-> class-or-object
      get-class-dwim
      .getName
      (str/split #"\.")
      last
      inf/underscore
      keyword))

(defn pk-val [entity]
  (get entity (pk entity)))

;;;
;;; Hooks
;;;

(defmulti pre-seek  get-class-dwim)
(defmulti post-seek get-class-dwim)

(defmulti pre-save  get-class-dwim)
(defmulti post-save get-class-dwim)

(defmulti pre-delete  get-class-dwim)
(defmulti post-delete get-class-dwim)


;; default impls

(defmethod pre-seek  :default [_] identity)
(defmethod post-seek :default [_] identity)

(defmethod pre-save  :default [_] identity)
(defmethod post-save :default [_] identity)

(defmethod pre-delete  :default [_] identity)
(defmethod post-delete :default [_] identity)


;; Helpers to write hooks

(defn wrap-pre-save-modifier
  "A helper to write a pre-save hook"
  [modifier]
  (fn [dml]
    (cond
      (contains? dml :values)
      (update-in dml [:values] (partial map modifier))

      (contains? dml :set)
      (update-in dml [:set] modifier)

      :else dml)))


;;;
;;; DML helpers
;;;

(defn update* [entity]
  (let [id (pk-val entity)]
    (assert id)
    (-> (update (table entity))
        (setv entity)
        (where (list '= 'id id)))))

(defn create* [entity]
  (-> (insert-into (table entity))
      (values [entity])))

(defn save* [entity]
  (if (pk-val entity)
    (update* entity)
    (create* entity)))

(defn delete* [entity]
  (let [id (pk-val entity)]
    (assert id)
    (-> (delete-from (table entity))
        (where (list '= 'id id)))))

(defn save [entity]
  (-> entity
      save*
      ((pre-save entity))
      m/execute!
      ((post-save entity))
      first
      vals
      first))

(defn delete [entity]
  (-> entity
      delete*
      ((pre-delete entity))
      m/execute!
      ((post-delete entity))
      first))


;;;
;;; Seeker
;;;

(defmacro seek'* [klass & modifiers]
  `(-> (select :*)
       (from (table ~klass))
       ~@modifiers
       ((pre-seek ~klass))))

(defmacro seek' [klass & modifiers]
  `(-> (seek'* ~klass ~@modifiers)
       m/query
       ((post-seek ~klass))))

(defn expand-query [q]
  (cond
    (nil? q)
    [:$= [1 1]]

    (map? q)
    (expand-query (map expand-query q))

    (sequential? q)
    (let [[k v] q]
      (cond
        (not (keyword? k))
        (if (= 1 (count q))
          (expand-query (first q))
          [:$and (map expand-query q)])

        (= k :$and)
        (expand-query v)

        (= k :$or)
        (cons :$or (rest (expand-query v)))

        (= k :$not)
        [:$not (expand-query v)]

        (re-find #"^\$" (name k))
        q

        ;; syntax sugars

        (and (string? v) (re-find #"%" v))
        [:$like [k v]]

        (instance? java.util.regex.Pattern v)
        [:$regex [k (str v)]]

        (vector? v)
        [:$in [k v]]

        :else
        [:$= q]))

    :else
    (throw (java.lang.IllegalArgumentException. (format "Cannot expand a query - %s" q)))))

(defmulti handle-query-op first)

(defmulti handle-option (fn [q [k v]] k))

(defn handle-options [q opts]
  (reduce handle-option q opts))

(defn seek* [klass & [query-map opts]]
  (-> (select :*)
      (from (table klass))
      (where (handle-query-op (vec (expand-query query-map))))
      (handle-options opts)
      ((pre-seek klass))))

(defn seek [klass & [query-map opts]]
  (-> (seek* klass query-map opts)
      m/query
      ((post-seek klass))))

(defn seek-one [klass & [query-map opts]]
  (first (seek klass query-map (assoc opts :limit 1))))

(defn count-by [klass & [query-map opts]]
  (-> (seek* klass query-map opts)
      (replace-select '(count :*))
      m/query
      first
      vals
      first))

;; query ops

(defmethod handle-query-op :$and [[op v]]
  `(~'and ~@(map handle-query-op v)))

(defmethod handle-query-op :$or [[op v]]
  `(~'or ~@(map handle-query-op v)))

(defmethod handle-query-op :$not [[op v]]
  `(~'not ~(handle-query-op v)))

(defmacro op-mapper [ops]
  `(do
     ~@(for [op ops]
         `(defmethod handle-query-op ~(keyword (str "$" (name op))) [[~'op [~'k & ~'args]]]
            (apply list (symbol ~(name op)) (if (keyword? ~'k) (symbol (name ~'k)) ~'k) ~'args)))))

(op-mapper [:= :!= :not= :is :is-not :in :not-in :like :not-like :regex :regexp :<>
            :< :<= :> :>= :between :not-between])

;; query options
;;  TODO: add more options

(defmethod handle-option :limit [q [_ v]]
  (limit q v))

(defmethod handle-option :offset [q [_ v]]
  (offset q v))

(defmethod handle-option :order-by [q [_ v]]
  (if (and (sequential? v)
           (not (#{:asc :desc} (last v))))
    (apply order-by q v)
    (order-by q v)))
