(ns materia.services.migration.service
  (:require [materia.services.migration.core :as core]
            [materia.utils :as u]
            [puppetlabs.trapperkeeper.core :refer [defservice]]
            [puppetlabs.trapperkeeper.services :refer [service-context]]
            [taoensso.timbre :as log]))

(defprotocol MigrationService
  (migrate [this])
  (remigrate [this])
  (rollback [this opts])
  (tag [this tag-name])
  (generate [this description])
  (get-changelog [this]))

(defservice migration-service
  MigrationService
  [[:ConfigService get-in-config]
   [:DatabaseService get-datasource]]
  (init [this context]
        (log/info "Initializing migration service")
        context)
  (start [this context]
         context)
  (stop [this context]
        (log/info "Shutting down migration service")
        context)
  (get-changelog [this]
                 (or (when-let [v (get-in-config [:migration :changelog])]
                       (-> v
                           u/parse-ns-var!
                           (->> (apply u/resolve-ns-var!))
                           var-get))
                     (when-let [path (get-in-config [:migration :path])]
                       (core/build-changelog path))))
  (migrate [this]
           (if-let [chlog (get-changelog this)]
             (do (core/update {:datasource (get-datasource)
                               :changelog chlog})
                 ;; FIXME: Always print success message regardless of
                 ;; whether new migration files exist or not.
                 (log/info (log/color-str :blue "Successfully migrated")))
             (log/info (log/color-str :yellow "No migrations to apply"))))
  (remigrate [this]
             (if-let [chlog (get-changelog this)]
               (do (core/reupdate {:datasource (get-datasource)
                                   :changelog chlog})
                   (log/info (log/color-str :blue "Successfully remigrated")))
               (log/info (log/color-str :yellow "No migrations to apply"))))
  (rollback [this opts]
            (if-let [chlog (get-changelog this)]
              (do (core/rollback (merge {:datasource (get-datasource)
                                         :changelog  (get-changelog this)}
                                        opts))
                  (log/info (log/color-str :blue "Successfully rolled back")))
              (log/info (log/color-str :yellow "No migrations to roll back"))))
  (tag [this tag-name]
       (core/tag {:datasource (get-datasource)
                  :tag tag-name}))
  (generate [this description]
            (when-let [dir (get-in-config [:migration :path])]
              (core/generate-empty-file dir description))))
