(ns materia.utils
  (:import [java.io FileNotFoundException]
           java.util.Properties))

(defn deep-merge
  [& vals]
  (if (every? map? vals)
    (apply merge-with deep-merge vals)
    (last vals)))

(defn deep-merge-with
  [f & maps]
  (apply
   (fn m [& maps]
     (if (every? map? maps)
       (apply merge-with m maps)
       (apply f maps)))
   maps))

(defn fmap [f m]
  (into {} (for [[k v] m] [k (f v)])))

(defn fmap-key [f m]
  (into {} (for [[k v] m] [(f k) v])))

(defn parse-ns-var! [s-ns-var]
  (if-let [[_ s-ns s-var] (re-matches
                           #"^([a-zA-Z0-9\.\-]+)/([a-zA-Z0-9\.\-]+)$"
                           s-ns-var)]
    [s-ns s-var]
    (throw (IllegalArgumentException.
            (str "Invalid format:\n\n\t"
                 s-ns-var
                 "\n\ns-ns-var must be of the form: '<namespace>/<var-name>'.")))))

(defn resolve-ns-var! [s-ns s-var]
  (let [sym-ns  (symbol s-ns)
        sym-var (symbol s-var)]
    (try (require sym-ns :reload)
         (catch FileNotFoundException e
           (throw (IllegalArgumentException.
                   (format "Unable to load var: %s/%s" s-ns s-var)))))
    (or (ns-resolve sym-ns sym-var)
        (throw (IllegalArgumentException.
                (format "Unable to load var: %s/%s" s-ns s-var))))))

(defmacro conj-> [expr & clauses]
  (let [pstep (fn [[test step]] [test `(conj ~step)])]
    `(cond-> ~expr
       ~@(apply concat (map pstep (partition 2 clauses))))))

(defn dissoc-in
  "Dissociates an entry from a nested associative structure returning a new
  nested structure. keys is a sequence of keys. Any empty maps that result
  will not be present in the new structure."
  [m [k & ks :as keys]]
  (if ks
    (if-let [nextmap (get m k)]
      (let [newmap (dissoc-in nextmap ks)]
        (if (seq newmap)
          (assoc m k newmap)
          (dissoc m k)))
      m)
    (dissoc m k)))

(defn comp*
  "Takes a set of functions and returns a fn that is the composition
  of those fns.  The returned fn takes a variable number of args,
  applies the leftmost of fns to the args, the next
  fn (left-to-right) to the result, etc.
  Also see the right-to-left version `comp`."
  ([] identity)
  ([f] f)
  ([f g]
   (fn
     ([] (g (f)))
     ([x] (g (f x)))
     ([x y] (g (f x y)))
     ([x y z] (g (f x y z)))
     ([x y z & args] (g (apply f x y z args)))))
  ([f g h]
   (fn
     ([] (h (g (f))))
     ([x] (h (g (f x))))
     ([x y] (h (g (f x y))))
     ([x y z] (h (g (f x y z))))
     ([x y z & args] (h (g (apply f x y z args))))))
  ([f1 f2 f3 & fs]
   (let [fs (list* f1 f2 f3 fs)]
     (fn [& args]
       (loop [ret (apply (first fs) args) fs (next fs)]
         (if fs
           (recur ((first fs) ret) (next fs))
           ret))))))

(defn partial*
  ([f] f)
  ([f arg1]
   (fn [arg] (f arg arg1)))
  ([f arg1 arg2]
   (fn [arg] (f arg arg1 arg2)))
  ([f arg1 arg2 arg3]
   (fn [arg] (f arg arg1 arg2 arg3)))
  ([f arg1 arg2 arg3 & more]
   (fn [arg] (apply f arg arg1 arg2 arg3 more))))

(defmacro fn-> [& body]
  `#(-> % ~@body))

(defmacro fn->> [& body]
  `#(->> % ~@body))

(defn map->properties [m]
  (reduce
   (fn [r [k v]]
     (cond
       (or (keyword? k)
           (symbol? k)) (.put r (name k) (str v))
           :else (.put r (str k) (str v)))
     r)
   (Properties.)
   m))

(defn crc32-hex [s]
  (let [c (java.util.zip.CRC32.)]
    (.update c (.getBytes (name s)))
    (format "%X" (.getValue c))))
