(ns materia.services.db.core
  (:refer-clojure :exclude [find])
  (:require [clojure.string :as str]
            [inflections.core :as inf]
            [jdbc.core :as jdbc]
            [jdbc.transaction :as tx]
            [stch.sql.format :as fmt]
            [stch.sql :refer :all]
            [taoensso.timbre :as log]))

(def ^:dynamic *current-db* nil)
(def ^:dynamic *current-conn* nil)
(def ^:dynamic *print-sql?* false)

(defmacro with-db [db & body]
  `(jdbc/with-connection [conn# ~db]
     (binding [*current-db* ~db
               *current-conn* conn#]
       ~@body)))

(defmacro with-transaction [& body]
  `(let [conn# *current-conn*]
     (tx/with-transaction conn#
       ~@body)))

(defn- print-sql [sql]
  (log/debug (prn-str sql)))

(defn- print-sql-dwim [sql]
  (when *print-sql?*
    (print-sql sql)))

(defn query
  ([q]
   (query q {:identifiers (comp str/lower-case inf/dasherize)}))
  ([q opt]
   (let [sql (fmt/format q :quoting :mysql)] ; TOOD: Make quoting style configurable
     (print-sql-dwim sql)
     (jdbc/query *current-conn* sql opt))))

(defn execute!
  ([q]
   (execute! q {:returning :all}))
  ([q opt]
   (let [sql (fmt/format q)]
     (print-sql-dwim sql)
     (jdbc/execute-prepared! *current-conn* sql opt))))

(defmacro with-sql-logger [& body]
  `(binding [*print-sql?* true]
     ~@body))

(defn wrap-sql-logger [handler]
  (fn [req]
    (binding [*print-sql?* true]
      (handler req))))


;;; SQL helpers (broken)

(defmulti pre-find (fn [query entity] entity))

(defmethod pre-find :default [query entity]
  query)

(defmulti post-find (fn [result entity] entity))

(defmethod post-find :default [result entity]
  result)

(defmulti pre-create (fn [dml entity] entity))

(defmethod pre-create :default [dml entity]
  dml)

(defmulti post-create (fn [result entity] entity))

(defmethod post-create :default [result entity]
  result)

(defmulti pre-update (fn [dml entity] entity))

(defmethod pre-update :default [dml entity]
  dml)

(defmulti post-update (fn [result entity] entity))

(defmethod post-update :default [result entity]
  result)


(defmulti pre-delete (fn [dml entity] entity))

(defmethod pre-delete :default [dml entity]
  dml)

(defmulti post-delete (fn [result entity] entity))

(defmethod post-delete :default [result entity]
  result)

(defmacro find [entity & modifiers]
  `(-> (select :*)
       (from ~entity)
       ~@modifiers
       (pre-find ~entity)
       query
       (post-find ~entity)))

(defn with-timestamp [e]
  (let [d (java.sql.Timestamp. (System/currentTimeMillis))]
    (-> e
        (update-in [:created-at] #(or % d))
        (assoc-in [:updated-at] d))))

(defmacro create [entity vals]
  `(-> (insert-into ~entity)
       (values ~vals)
       (pre-create ~entity)
       execute!
       (post-create ~entity)))

(defmacro update-entity [entity val]
  `(let [id# (:id ~val)]
     (assert id# "'id' key is required")
     (-> (update ~entity)
         (setv (dissoc ~val :id))
         (where (list (quote =) (quote id) id#))
         (pre-update ~entity)
         execute!
         (post-update ~entity))))

(defmacro delete [entity val]
  `(let [id# (:id ~val)]
     (assert id# "'id' key is required")
     (-> (delete-from ~entity)
         (where (list (quote =) (quote id) id#))
         (pre-delete ~entity)
         execute!
         (post-delete ~entity))))
