(ns series.csv
  (:require
    [clojure.data.csv :as csv]
    [clojure.string :as str]
    [clojure.java.io :as io]
    [clj-time.core :as t]
    [clj-time.format :as fmt]
    [taoensso.tufte :as tufte :refer (defnp p profiled profile)]
  ))

(def base-directory "../DAILY/")


(def row-date-format-
  (fmt/formatter "M/d/yyyy H:m:s a")) ; 1/31/1990 12:00:00 AM

(defn parse-row-
  [[date-str open-str high-str low-str last-str vol-str]]  ; date,PX_OPEN,PX_HIGH,PX_LOW,PX_LAST,PX_VOLUME
  {:date (fmt/parse row-date-format- date-str)
   :close (Float/parseFloat last-str)})


(defn csv-data->maps [csv-data]
     (doall  (map zipmap
          (->> (first csv-data) ;; First row is the header
               (map keyword) ;; Drop if you want string keys instead
               repeat)
   	  (rest csv-data))))


(defn non-empty-string?- [v]
  (if (instance? String v) (not (clojure.string/blank? v)) false))

(defn float-or-nil- [str]
  (if (non-empty-string?- str) (Float/parseFloat str) nil)
  )

(defn read-csv [x]
  (p :read-csv (csv/read-csv x)) )

(defn csv-data-maps [x]
  (p :csv-maps (csv-data->maps x)) )

(defn sanitize [list]
  (map (fn [csv-record]
         ( -> csv-record
              (update :date #(fmt/parse row-date-format- %))
              (clojure.set/rename-keys {:PX_LAST :close
                                        :PX_HIGH :high
                                        :PX_LOW :low
                                        :PX_OPEN :open
                                        :PX_VOLUME :volume
                                        })

              (update  :open #(float-or-nil- %))
              (update  :high #(float-or-nil- %))
              (update  :low #(float-or-nil- %))
              (update  :close #(float-or-nil- %))
              (update  :volume #(float-or-nil- %))
              )) list)
  )

(defn sanitize-list [list]
  (p :csv-sanitize (sanitize list)))


(defn exists [filename]
  (p :csv-exists (.exists (io/as-file filename))))


(defn remove-nil [list]
  (p :csv-remove-nil
     (remove #(nil? (:PX_LAST %)) list)
     ))

(defn remove-empty [list]
  (p :csv-remove-empty
     (remove #(= (:PX_LAST %) "") list) )
     )





(defn load-csv-
  [folder mySymbol]
  ( let [filename (str folder mySymbol ".csv")]
    (if (exists filename)
       (->> (p :csv-slurp (doall (slurp filename)) )
            (read-csv)
            (csv-data-maps)
            ;(drop 1) ; drop header row
            (remove-nil)
            (remove-empty)
            (sanitize-list)
           )
            ;(map parse-row-))
       (do (println "no csv file found for: " mySymbol)
           nil
        ))))





(defn filter-start-date [series date-start]
    (let [is-after (fn [bar] (t/after? (:date bar) date-start ))]
    (filter is-after series)))


(defn filter-since-1990 [series]
  (p :csv-filter (doall (filter-start-date series (t/date-time 1989 12 31))) ) )

(defn load-series [symbol]
  (p :csv-load (doall (load-csv- base-directory symbol) )  ))


(defn available-range [symbol]
  (let [data (load-csv- base-directory symbol)
        start (first data)
        end (last data)]
    {:start (:date start)  :end (:date end)}
    ))


; UNIVERSE *****************************************

(defn symbols-with-timeseries []
  ( ->> (mapv str (filter #(.isFile %) (file-seq (clojure.java.io/file base-directory))))
        ;(map str/split)
        (map #(str/split % #"/"))
        (map last)
        (map #(str/split % #".csv"))
        (map first)
        (vec)
      ))



; WRITE *****************************************


;(with-open [reader (io/reader "in-file.csv")]
;  (doall
;    (csv/read-csv reader)))

(defn write-csv [symbol series]
  (let [columns [:date :PX_OPEN :PX_HIGH :PX_LOW :PX_LAST :PX_VOLUME]
        headers (map name columns)
        rows (mapv #(mapv % columns) series)]
    (with-open [writer (io/writer "out-file.csv")]
       (csv/write-csv writer (cons headers rows) ))))

(comment

  ; date,PX_OPEN,PX_HIGH,PX_LOW,PX_LAST,PX_VOLUME
  ; 4/14/2011 12:00:00 AM,19,19,19,19,
  ; 4/15/2011 12:00:00 AM,19.1,21.7,19.1,21.7,4124893
  ; 4/18/2011 12:00:00 AM,21.28,21.9,20.755,21,469876


  (write-csv "FLO Test" [
    {:date (t/date-time 2011 04 14) :PX_OPEN 19 :PX_HIGH 19 :PX_LOW 19 :PX_LAST 19 :PX_VOLUME nil}
    {:date (t/date-time 2011 04 15) :PX_OPEN 19.1 :PX_HIGH 21.7 :PX_LOW 19.1 :PX_LAST 21.7 :PX_VOLUME 4124893}
    {:date (t/date-time 2011 04 18) :PX_OPEN 21.28 :PX_HIGH 21.9 :PX_LOW 20.755 :PX_LAST 21 :PX_VOLUME 469876}
  ])





  (clojure.string/escape "I want 1 < 2 as HTML, & other good things."
                         {\< "&lt;", \> "&gt;", \& "&amp;"})


  (fmt/show-formatters)

  (float-or-nil- "34.34")

  (def symbol-
  ;"DAX Index"
    "MCD UN Equity"
  ;"ADS GY Equity"
    )

  (def test-
    (load-series symbol-))

; evaluate first 5 elements of the lazy list
  (take 5 test-)
  (doall test-)

  (take 5 (filter-since-1990 test-))

  (profile {} (p :csv (do (load-series "AAPL US Equity") nil )))

  (count (load-series "AAPL US Equity"))


  (profile {} (p :slurp (doall (slurp "../DAILY/AAPL US Equity.csv")) ))

  (available-range "MSFT US Equity")

; UNIVERSE *****************

  (symbols-with-timeseries)
  (count (symbols-with-timeseries))

  (let [x (symbols-with-timeseries)]
    (for [s x]
      (println "* " s) ))

  (map println (symbols-with-timeseries))
  (type (symbols-with-timeseries))





  ) ; ***************************************************************
