(ns series.core
  (:require
    [clojure.string :as str]
    [clojure.java.io :as io]
    [clj-time.core :as t]
    ;[series.csv :as ts-csv]
    [mongo.series] ; needs to be like this; direct referring fucks up function names in this module

    [data.equity-symbol :refer [categorize-symbol]]
    [series.splits :refer [timeseries-split-adjust load-split-data timeseries-split-adjust-via-splits]]

    [series.compress :refer [compress group-month]]
  ))

(defn available-range [symbol]
   {:symbol symbol :start nil :end nil}
  )

(defn symbols-with-timeseries []
   []
  )

(defn load-series-
  "Loads a series from db
   Either full series, or interval [start-end]"
  [params symbol]
  (let [start (get-in params [:timeframe :start])
        end (get-in params [:timeframe :end])]
    (if (nil? start)
        (mongo.series/load-series symbol)
        (mongo.series/load-series-partial symbol start end) )))

(defn do-splits?-
  "returns true if we have to run split-adjustment calculation factor
   params/symbol. false otherwise"
  [params symbol]
  (let [split-adjust (:split-adjust params)
        category (categorize-symbol symbol)]
    (and (or (nil? split-adjust) split-adjust)
         (= (:category category) "Equity"))))

(defn split-adjust-
  "split adjusts a series (if wanted and sensible for the category)
   unchanged if not.
  "
  [params symbol series]
  (if (do-splits?- params symbol)
      (let [split-data (get-in params [:data :splits])]
        (if (nil? split-data)
            (timeseries-split-adjust series symbol)
            (timeseries-split-adjust-via-splits series split-data)))
      series))

(defn compress-
  "compresses a series to monthly or leaves daily series"
  [params series]
  (case (:frequency params)
    :daily series
    :monthly (compress group-month series)
    [] ))


(defn load-series-params [params symbol]
  "loads a timeseries for a symbol
    source is always daily data
    does adjustments if wanted/necessary
      - compression level wanted [:daily :monthly]
      - split-adjustments

   parameter:
     :timeframe
          :start
          :end
     :frequency :daily :monthly
     :split-adjust  true/false
     :data
         :splits
   If any data item is specified, instead of going to db will use this data
  "
  (->> symbol
       (load-series- params)
       (split-adjust- params symbol)
       (compress- params)))


(defn load-series
  "wrapper for compatibility (see load-series-params)"
   ([symbol]
     (load-series-params
        {:split-adjust true :frequency :daily} symbol))
   ([symbol frequency]
     (load-series-params
       {:split-adjust true :frequency frequency} symbol))
    ([symbol frequency start end]
     (load-series-params
       {:split-adjust true :frequency frequency :timeframe {:start start :end end}} symbol)) )


(comment

;"EBS AV Equity"
;"MSFT US Equity"
;(load-series :daily)
(future (-> "BP/ LN Equity"
    (load-series  :monthly)
    (#(take 5 %))
    (println)))

(last (load-series "DAX Index" :monthly))
(load-series "BP/ LN Equity" :monthly)

(load-series "MUV2 GR Equity" :monthly (t/date-time 1997 6 1) (t/date-time 1997 9 30))

)
