(ns mongo.portfolio
  (:require
    [clojure.tools.logging :refer [info]]
    [clj-time.core :as t]
    [clojure.string :as str]
    [monger.core :as mg]
    [monger.collection :as mc]
    [monger.joda-time]
    [mongo.db :refer [db]] ))

(defn update-member-count- [portfolio-symbol]
  (let [instrument (mc/find-one-as-map db "instruments" {:symbol portfolio-symbol})
        member-count (count (:members instrument))  ]
     (mc/update db "instruments" {:symbol portfolio-symbol}
                                {:$set {:COUNT_INDEX_MEMBERS member-count}} {:upsert false})
      {:symbol portfolio-symbol :count member-count}
    ))

; Add/Remove (used by Web UI to add/remove members to a portfolio)

(defn portfolio-symbol-add [portfolio-symbol instrument-symbol]
    (mc/update db "instruments"  {:symbol portfolio-symbol} {:$addToSet {:members instrument-symbol}})
    (update-member-count- portfolio-symbol)
    nil)

(defn portfolio-symbol-remove [portfolio-symbol instrument-symbol]
  (mc/update db "instruments"  {:symbol portfolio-symbol} {:$pull {:members instrument-symbol}})
  (update-member-count- portfolio-symbol)
  nil )

(defn set-list [symbol members]
    (mc/update db "instruments"
               {:symbol symbol}
               {:$set {:COUNT_INDEX_MEMBERS (count members)
                       :members members
                       :symbol symbol
                       }} {:upsert true}))

(defn list-add [symbol]
  (mc/update db "instruments"  {:symbol symbol} {:symbol symbol
                                                 :category "List"
                                                 :members []} {:upsert true})
  (update-member-count- symbol)
  nil)


(defn list-members [symbol]
  (->> {:symbol symbol}
       (mc/find-one-as-map db "instruments")
       (:members)))



(defn unwind-members [instrument]
  (assoc instrument :members
       (vec (flatten (:members instrument)))))




(defn list-detailed-raw [symbol]
 (mc/aggregate db "instruments"
   [{:$match {:symbol symbol}}
    {:$unwind "$members"}  ; 2019 09 fh: this is needed due to the mongodb lookup bug.
    {:$lookup {:from         "instruments"
               :localField   "members" ; 2019 09 fh: this should work, but due to mongo bug, lookup does not work on an array
               :foreignField "symbol"
               :as           "details"}}
    {:$project {"_id"                 0
                "symbol"              1
                "NAME"                1
                "description"         1
                "INDX_SOURCE"         1
                "COUNT_INDEX_MEMBERS" 1
                "details.symbol"            1
                "details.name"              1}}
    {:$group {:_id  {:symbol "$symbol"}
              :description         {:$first "$description"}
              :INDX_SOURCE         {:$first "$INDX_SOURCE"}
              :NAME                {:$first "$NAME"}
              :COUNT_INDEX_MEMBERS {:$first "$COUNT_INDEX_MEMBERS"}
              :members             {:$push "$details"}}}
     ]))

(defn list-detailed [symbol]
  (-> symbol
      (list-detailed-raw)
      (first)
      (unwind-members)))


(comment
   (portfolio-symbol-add "test List" "AAA AV Equity")
   (portfolio-symbol-remove "test List" "AAA AV Equity")
   (set-list "willy List" ["A Equity" "B Equity" "C Index"])
   (list-add "billabongo List")

   (list-detailed "florian List")

  )
