(ns mongo.instrument
  (:refer-clojure :exclude [sort find])
  (:require
    [clojure.tools.logging :refer [info]]
    [clj-time.core :as t]
    [clojure.string :as str]
    [monger.core :as mg]
    [monger.collection :as mc]
    [monger.joda-time]
    [monger.query :refer [with-collection paginate fields find sort]]
    [mongo.db :refer [db]]
    ))

(defn load-symbol
  ([symbol]
    (mc/find-one-as-map db "instruments" {:symbol symbol}))
  ([symbol fields]
    (mc/find-one-as-map db "instruments" {:symbol symbol} fields)))

;(clojure.repl/doc mc/find-one-as-map)

(defn load-symbol-safe [symbol & fields]
  (let [inst (load-symbol symbol)]
    (-> inst
        (dissoc inst :_id)
        ;(println)
        )))

(defn load-equities []
  (mc/find-maps db "instruments" {:category "Equity"}))

(defn update-instrument [instrument]
  (mc/update db "instruments" {:symbol (:symbol instrument)} {:$set instrument} {:upsert true}))

(defn save-splits [symbol splits]
  (mc/update db "instruments" {:symbol symbol} {:$set splits} {:upsert false}))


(defn search-instrument [query category exchange limit]
  (let [query {:$or [ {:name {:$regex query :$options "i"}}
                      {:symbol {:$regex query :$options "i"}}]}
                      
       query (if (= category "") query (assoc query :category category))
       query (if (= exchange "") query (assoc query :exchange exchange))]
    ;(mc/find-maps db "instruments" query
    ;; find top 10 scores
    (with-collection db "instruments"
      (find query)
      ;(fields [:symbol :name :exchange :category]) ;
      (fields {:symbol-only 1 :symbol 1 :name 1 :exchange 1 :category 1 :_id 0})
      (sort {:name 1})
      (paginate :page 1 :per-page limit)
      )))


(defn set-name [instrument]
  (if (nil? (:name instrument)) (assoc instrument :name "")  instrument))

(defn sanitize-name [instruments]
  (map set-name instruments))



(defn load-changed-symbols []
  (mc/find-maps db "instruments" 
                {:MARKET_STATUS "TKCH"}
                {:symbol 1
                 :desnotes 1
                 :_id 0}))

(defn change-symbol [symbol-old symbol-new]
  (mc/update db "instruments" 
             {:symbol symbol-old} 
             {:$set {:symbol symbol-new}}
              {:upsert false}))


(comment

  (set-name {:a 5})
  (sanitize-name [{:name "a" :x 1} {:x 2}])
  (clojure.pprint/print-table (search-instrument "DAX" "Index" "" 10))
  (clojure.pprint/print-table (sanitize-name (search-instrument "BP" "Equity" "" 10)))


  (search-instrument "con" "Equity" "" 10)



  (load-symbol "DAX Index")
  (load-symbol "DAX Index" [:name :symbol])
  (load-symbol "MCD US Equity")

  (save-splits "SXNP Index" {:super "man" :bongo "bingo" :dvd [{:a 3} {:a 5}]})

  (count (load-equities))

  (update-instrument db {:symbol "MCD US Equity" :bongo "123" :xx (t/date-time 1986 10 30)})

  (load-changed-symbols)
  (change-symbol "ALA-U CN Equity" "ALA CN Equity")
  
  )
