(ns job.workflow
  (:require
    [mongo.db :refer [db]]
    [job.run :refer [run index symbols]]

    [import.series]
    [data.equity-symbol :refer [separate-category]]

    ; Category Dependent
    [import.index]
    [import.bond]
    [import.currency]
    [import.equity]
    [import.splits]
    [import.report-individual]
    [import.report-summary]

    ))

(defn add-cost [job-results]
  (apply merge job-results))

(defn run-item- [f text symbol]
  (let [info #(println (str "[" %1 "]") %2 " .. ")]
    (info symbol text)
    (f symbol)))

(defn run-workflow [workflow symbol]
  (->> (map #(run-item- (get % 0) (get % 1) symbol) workflow )
       (add-cost)))


(def equity-workflow
  [[import.equity/equity-descriptive? "Equity Description"]
    [import.splits/splits? "Split Status (if needed)"]
    [import.report-summary/financial-summary? "FinancialReport Summary (if needed)"]
    [import.report-individual/financial-reports "IndividualFinancialReports"]
    ;[scrubbing-reports/scrub-reports "scrubbing reports (data migration) [NOT IMPLEMENTED]"]
    ;[missing-field/load-missing-report-fields "load missing fields (data migration) [NOT IMPLEMENTED]"]
  ])

(def series-workflow
  [[(partial import.series/daily-series-if-empty "19500101" "20201231") "EOD Series"]
   [import.series/daily-series-latest "EOD Series Update"]])

(def series-reload-workflow
  [[(partial import.series/daily-series "19500101" "20201231") "EOD Series"]
   ])


(def bond-workflow
  [[import.bond/bond-info? "Bond Info"]])

(def index-workflow
  [[import.index/index-info? "Index Info"]])

(def currency-workflow
    [[import.currency/currency-info? "Currency Info"]])


(defn unknown-category [symbol]
  {:cost 0 :unknown-category 1})

(def unknown-workflow
  [[unknown-category "Unknown Category"]])


(defn run-workflow-category [symbol]
  (let [category (:category (separate-category {:symbol symbol}))]
    (run-workflow
      (case category
            "Equity" equity-workflow
            "Corp" bond-workflow
            "Index" index-workflow
            "Curncy" currency-workflow
            unknown-workflow)
      symbol)))

(defn series-only [symbol]
    (run-workflow series-workflow symbol))

(defn financial-workflow [symbol]
  (add-cost
    [(run-workflow series-workflow symbol)
     (run-workflow-category symbol) ]))


(comment

  ; TEST INDIVIDUAL FUNCTINS
  (run-workflow equity-workflow "BP/ LN Equity")

  ; TEST INFO
  (run-workflow-category "BP/ LN Equity")
  (run-workflow-category "AU8162573 Corp")
  (run-workflow-category "DAX Index")

  ; TEST SERIES
  (run-workflow series-workflow "BP/ LN Equity")
  (run-workflow series-workflow "AU8162573 Corp")
  (run-workflow series-workflow "AN288649 Corp") ; unknown symbol
  (run-workflow series-workflow "LW1230900 Corp")


  ; RUN - INDIVIDUAL ITEMS
  (run series-only "bond List" index)
  (run run-workflow-category "bond List" index)


  (run series-workflow "BP/ LN Equity")
  
  

  ;; WORKFLOW
  (financial-workflow "BP/ LN Equity")
  (financial-workflow "DAX Index")
  (financial-workflow "LW6890732 Corp")

  ; FLORIAN
  (run financial-workflow "florian List" index)
  (run financial-workflow "bond List" index)
  (run financial-workflow "currencies List" index)
  (future (run financial-workflow "indices_many List" index))

  (future (run financial-workflow "bond List" index))
  (future (run financial-workflow "DAX Index" index))


  ; REIT
  (financial-workflow "MAC US Equity")
  (financial-workflow "TNT-U CN Equity")
  (financial-workflow "UQA AV Equity")
  (financial-workflow "BP US Equity")
  (financial-workflow "CAI AV Equity")
  (run financial-workflow "LNA GY Equity")

  ; europe
  (run financial-workflow "ATX Index" index)
  (run financial-workflow "DAX Index" index)
  (run financial-workflow "HDAX Index" index)
  (future (run financial-workflow "BE500 Index" index))
  ; STOXX 600
  
  ; usa
  (run financial-workflow  "SPX Index" index)
  (run financial-workflow "RIY Index" index)                ; Russel 1000
  (run financial-workflow "RTY Index" index)                ; Russell 2000
  (run financial-workflow "RAY Index" index)                ; Russell 3000
  
  ;sector
  (run financial-workflow "BWIRON Index" index)
  (run financial-workflow "BWAUTM Index" index)
  (run financial-workflow "BBRSRGLD Index" index)

  (run financial-workflow "BWORLDPR Index" index)
  (run financial-workflow "BWORLDEU Index" index)

  ; S&P 500
  (run financial-workflow "SXXP Index" index )


  (run-workflow series-reload-workflow "UQA AV Equity")


  ;DATE_OR_YEAR_OF_INCORPORATION
  ;EQY_FISCAL_YR_END
  ; END OF FISCAL YEAR.
  ; LATEST_PERIOD_END_DT_FULL_RECORD
  

  )
