(ns fundamental.table
  (:require
    [clojure.string :as str]
    [clj-time.core :as t]
    [clojure.algo.generic.functor :only (fmap)]
    [mongo.report :refer [load-reports-for-symbol get-report-fields]]
    [fundamental.report :refer [yearly-report?]]
    ))

; REPORT TABLE

(defn get-field-history- [reports field]
  (map #(get-in % [:fields field]) reports))

(defn report-table
  "multi-year + multi-fields financial report table"
  ([fields reports]
  (let [periods (map :period reports)
        rows (map #(get-field-history- reports %) fields)
        rows-with-fieldheaders (zipmap fields rows)]
    (conj [{:period periods}] rows-with-fieldheaders)))
  ([symbol]
     (let [reports (load-reports-for-symbol symbol)
           reports-filtered (filter yearly-report? reports)
           reports-sorted (sort-by :period reports-filtered)
           fields (get-report-fields)
           report (report-table fields reports-sorted)]
       report)))

(defn report-table-field [symbol field]
  (let [reports (load-reports-for-symbol symbol)
        reports-filtered (filter yearly-report? reports)
        reports-sorted (sort-by :period reports-filtered)
        fields [field]
        report (report-table fields reports-sorted)]
    report))


(defn divide-tuple-
  "divide a by b, parameter is a tuple"
  [tuple]
  (let [a (first tuple)
        b (last tuple)
        ab-valid? (and (not (nil? a)) (not (nil? b)) (not (= 0 b)))
        ;xx (println "div:" a b )
        ]
        (if ab-valid? (/ a b) nil)))

(defn divide-by-shares- [row shares]
  (let [pairs (map vector row shares)]
     (vec (map divide-tuple- pairs))))

(defn report-per-share
  "converts a multi-year financial report table to per share data"
  [report]
  (let [periods (first report)
        data (last report)
        fields (keys data)
        shares (:BS_SH_OUT data)
        ;xx (println "shares: " (count shares))
        ;xx (println "data: " (count data))
        per-share-data (clojure.algo.generic.functor/fmap #(divide-by-shares- % shares) data)
       ]
       [periods per-share-data]
    ))



(comment

  (let [fields (get-report-fields)
        reports (load-reports-for-symbol "A2A IM Equity")
        periods (map :period reports)
        xx (println "Periods: " periods)
        first-report (first reports)
        ;xx (println "First Report Full: " first-report)
        xx (println "NetIncome first Report: " (:NET_INCOME (:fields first-report)))
        xx (report-table fields reports)
        xx (map #(get-in % [:fields :SALES_REV_TURN]) reports)
        ;xx (println (into vector (select-keys (:fields first-report) (vec fields))))
        ]
    )

  (divide-by-shares- [1000 2000 3000 4000] [100 100 100 200])

  ; the report table
  (-> "BP/ LN Equity"
      (report-table)
      (report-per-share)
      (last)
      ;(keys)
      (:BS_SH_OUT)
      (println)
      )




  ; print history of one field; good for debugging of available fields.
  (report-table-field "BP/ LN Equity" :NET_INCOME)
  (report-table-field "MCD UN Equity" :BS_SH_OUT)

  (map vector [nil nil 1 2 3 4 5] [nil nil 6 7 nil 9 0])


  )
