(ns fundamental.report
  (:require
    [clojure.tools.logging :refer [info]]
    [clojure.string :as str]
    [clj-time.core :as t]
    [mongo.report :refer [load-reports-for-symbol]]))


(defn yearly-report-symbol? [report-symbol] (str/ends-with? report-symbol ":A"))

(defn yearly-report? [report] (str/ends-with? (:period report) ":A"))


; *** Split 1999:A 2001:Q3  => type:Q year:2001 number:3

(defn parse-int [s]
  (Integer/parseInt s))


(defn extract-year-type-number [report]
  (let [;xx (println "extract: " report)
        period (:period report)
        regex-result (re-find #"(\d+):([A-Z])+(\d*)" period) ; vector, first elememnt entire regex
        ]
    (if (nil? regex-result) nil
                            (let [[_ year type number] regex-result
                                  year (parse-int year)
                                  ]
                              (if (str/blank? number)
                                {:type type :year year}
                                {:type type :year year :number (parse-int number)}
                                )
                              ))))

; *** ESTIMATED RELEASE DATE

(defn parse-int [s]
  (Integer/parseInt (re-find #"\A-?\d+" s)))

(defn estimated-release-date [report]
  (let [ period-type (:period report)
        [year period] (str/split period-type #":")
        year (parse-int year)
        ]
    (t/date-time (+ year 1) 6 1)))


(defn add-estimated-release-date-if-needed [report]
  (if (nil? (:releasedate report)) (assoc report :releasedate (estimated-release-date report)) report )
  )


; ** ESTIMATES REPORT END DATE


(defn report-end-date-estimated [report]
  (let [period-type (:period report)
        [year period] (str/split period-type #":")
        year (parse-int year)
        ]
    (t/date-time year  12 31)))

(defn report-end-date [report]
  (let [end-date (get-in report [:fields :LATEST_PERIOD_END_DT_FULL_RECORD])]
   (if (nil? end-date)
     (report-end-date-estimated report)
     end-date)))


; LOAD REPORTS

(defn sanitize-report [report]
 (-> report
     (dissoc :_id)
     (add-estimated-release-date-if-needed)))

(defn load-reports [symbol]
 (let [reports (load-reports-for-symbol symbol "A")
       reports-sanitized (map sanitize-report reports)
       reports-yearly (filter yearly-report? reports-sanitized)
       reports-yearly (remove #(nil? (:releasedate %)) reports-yearly)
       xx (println "reports all: " (count reports) "yearly-ok: " (count reports-yearly))
       ;xx (println "last report: " (last reports-yearly))
       ]
   reports-yearly))



(comment
  (re-find #"(\d+):(\w)+(\d*)" "1999:Q3")

  (extract-year-type-number {:period "2001:A"})
  (extract-year-type-number {:period "1999:Q3"})
  (extract-year-type-number {:period "2001-A"})             ; nil, because format was not correct


  (report-end-date {:period "2017:A"})
  (report-end-date {:period "2017:A" :fields {:LATEST_PERIOD_END_DT_FULL_RECORD (t/date-time 2018 05 13)}})


  (-> "MAC US Equity"
      (load-reports)
      (count))

  )
