(ns data.equity-symbol
  (:require
    [clojure.data.csv :as csv]
    [clojure.string :as str]
    [clojure.java.io :as io]
    [clj-time.core :as t]
    [clj-time.format :as fmt]
    [taoensso.tufte :as tufte :refer (defnp p profiled profile)]
      ))

; SYMBOL => TICKER EXCHANGE CATEGORY [MO US Equity]

(defn in?
  "true if coll contains elm"
  [coll elm]
  (some #(= elm %) coll))


(defn capitalize-category [data]
  (update data :category clojure.string/capitalize ))


(defn remove-empty-exchange [data]
  (if (clojure.string/blank? (:exchange data))
    (dissoc data :exchange)
    data
    ))

(defn array-to-map [array]
  (if-not (nil? array)
    (->> array
         (drop 1)
         (zipmap [:symbol-only :exchange :category ])
         (capitalize-category)
         (remove-empty-exchange)
         )))



(defn categorize-symbol [symbol]
  (->> symbol
       (re-matches #"([\w-\\*&//]+)\s*([\w]*)\s+(\w+$)")   ;  "(.*)\s(\w+)\s*"
       (array-to-map)))


(defn separate-category [instrument]
   (->> instrument
        (:symbol)
        (categorize-symbol)))



; FIND EQUITY COMPOSITE EXCHANGE

; Bloomberg Equity Exchanges is accessed on "EPRX" page.
; Many Equity Exchanges have a composite Exchange.
; In order to keep symbols low, we only deal with composite exchanges.

(defn csv-data->maps [csv-data]
  (doall  (map zipmap
               (->> (first csv-data) ;; First row is the header
                    (map keyword) ;; Drop if you want string keys instead
                    repeat)
               (rest csv-data))))

(defn load-exchanges []
  (->> "resources/EPRX.csv"
       (slurp)
       (csv/read-csv)
       (csv-data->maps)
       (map #(assoc % :EXCH (str/split (:EXCH %) #", ")) )
       ))

(defn build-composite-map [exchanges]
  (let [m (map #(select-keys % [:Comp :EXCH]) exchanges)
        m (remove #(= (:Comp %) "n/a") m)]
    m
    ))

(def composite-list (build-composite-map (load-exchanges)))


(defn equals-exchange [exchange-code exchange]
  (not= -1 (.indexOf (:EXCH exchange) exchange-code))
  )

(defn find-composite
  "for any exchange code, returns composite exchange code
   or nil (if no composite found)"
  [exchange-code]
  (let [exchange (some #(if (equals-exchange exchange-code %) %) composite-list)
        ;xx (println "found exchange: " exchange)
        ]
     (if (nil? exchange)
        nil
       (:Comp exchange))
    ))

(comment

  ; Index does not have an exchange
  (categorize-symbol "DAX Index")

  ; note the additional spaces
  (categorize-symbol "EBS  AV  Equity")
   (categorize-symbol "DAX      Index")

  ; note capitalization (e samall Y large)
  (categorize-symbol "EBS  AV  equitY")

  ; should fail - no category
  (categorize-symbol "asdf")

  ; symbol contains / - * &
  (categorize-symbol "FP/ LN Equity")
  (categorize-symbol "AIF-U CN Equity")
  (categorize-symbol "ARA* MM Equity")
  (categorize-symbol "PE&OLES* MM Equity")

  (separate-category {:symbol "DAX Index" :name "DAX"})
  (separate-category {:symbol "EBS  AV  Equity" :name "EBS"})


  (println composite-list)

  (find-composite "GY")
  (find-composite "GR")
  (find-composite "UN")
  (find-composite "US")

  )
