(ns analysis.app
 (:require
   [reagent.core :as r]
   [cljss.core :as css]

   [helper.main :refer [tap]]
   [comp.local-storage :refer [ls-get ls-set!]]
   [comp.mui :as mui]
   [comp.button :as cbutton]
   [comp.mytab :as mytab]
   [comp.combo :refer [combo]]

   [app.quant.db :refer [navigate-home]]

   [styles.analysis :refer [master menu strategy button-goto]]
   [analysis.combined :refer [combined-component]]

   ; Components that are being used by the analysis switcher
   [comp.highchart] ; for demo
   [seasonality.instsim :refer [simulate-instrument-app]]
   [financials.core]
   [tradingview.chart]
   [instrument.info :refer [instrument-info-page]]
   )
 (:require-macros
   [devcards.core :as dc :refer [defcard defcard-rg]]))

; local-storage key used
(defonce ls-key- "analysis")

(defonce state-analysis (r/atom {}))


(defn init []
  (->> ls-key-
       (tap "analysis-app: load state from local-storage")
       (ls-get)
       (tap "analysis-app: loaded state is:")
       ;(reset! state-analysis)
       )
  (r/track!
    (fn [] (println "analysis-app: save state to local-storage")
           (ls-set! ls-key- @state-analysis))))

(defn set-symbol [newsymbol]
  (do (.log js/console "analysis-app: symbol is: " newsymbol)
      ;(reset! symbol newsymbol)
      (swap! state-analysis assoc :symbol newsymbol)))

(defn set-strategy [strategy]
   (do (.log js/console "analysis-app: strategy is: " strategy)
       (swap! state-analysis assoc :strategy strategy)))

(defn empty-strategy [symbol]
  [:h1 "Empty Strategy for " symbol]
  )


(defn analysis [strategy-list]
  (let [rows (partition 2 strategy-list)
        strategy-list (map #(nth % 0) rows)
        view-list (map #(nth % 1) rows)]
    (fn []
      (let [symbol (:symbol @state-analysis)
            strategy-name (:strategy @state-analysis)
            strategy-name (if (nil? strategy-name) "demo" strategy-name)
            strategy-index (.indexOf strategy-list strategy-name)
            strategy-view (if (> strategy-index -1) (nth view-list strategy-index ) nil)
            ; _ (println "analysis-app: render strategy: " strategy-name)
            strategy-view (if (nil? strategy-view) empty-strategy strategy-view)
            ]
       [master
         [menu ; MENU (Strategy Combo + Instrument Selector)
            [button-goto {:on-click navigate-home} "<-"]
            [combo strategy-name strategy-list set-strategy]    ; Strategy selector
            [combined-component state-analysis]] ; Symbol Selector
         [strategy   ; Strategy View ; :<> ;#analysis-strategy  ; FRAGEMENT does not render to the DOM.
            [strategy-view symbol]]]
    ))))

; APPLICATION

(defn demo-view [symbol]
  [:h1 "demo" symbol])


(defn analysis-app []
   [analysis ["demo" demo-view
              "info" instrument-info-page
              "tradingview"  tradingview.chart/chart
              "seasonal" simulate-instrument-app
              "financials" financials.core/financials-page ]])


; DEV-CARDS

(defn demo-info- [symbol]
  [:<>
    [:h1 {:style {:background-color "lightblue"}} "INFO " symbol]
    [:p "hello world"]])

(defn demo-seasonal- [symbol]
  [:h1 "Seasonals for " symbol])

(defn demo-financials- [symbol]
  [:h1 "Financials for " symbol])

(defn demo-chart- [symbol]
  [:div
    [:h1 "Chart for " symbol]
    [comp.highchart/demo-chart]])

(defn teststrategy [symbol]
  [:h1 "TEST for symbol" symbol])

(defcard-rg strategy-selection-box
    "strategy selection box"
    [analysis [
        "info" demo-info-
        "chart"  demo-chart-
        "seasonal" demo-seasonal-
        "financials" demo-financials-
        ]])
