(ns mascip.debug
  (:require [clojure.string :as s]
            [puget.printer  :refer :all]
            [robert.hooke   :refer :all]
  ))

"Acknowledgement:
 - The (c) function is inspired by Micheal Feather's c function for Ruby, in his article \"Literate Chains for Functional Programming\"
 - (dbg) was found in the article \"Conditioning the REPL\" by John Lawrence Aspden"


;;;; (dbg) (and (pdbg)
(defmacro dbg
  "Try this:
  (+ (* 2 3) (dbg (* 8 9)))
  => dbg: (* 8 9) => 72"
  [x]
  `(let [x# ~x] (println "dbg:" '~x "=>" x#) x#))

(defmacro pdbg
  "Like (dbg) but prints with colors"
  [x]
  `(let [x# ~x] (do (println "--")(cprint '~x)(println "->")(cprint x#) (println "--") x#)))


;;;; (c) and (cs)

(defn c [data s]
  "A comment that helps you debug your -> thread chains. Is takes data in and passes the same data out. For example:
  (-> [1 2] (c \"The vector\") reverse (c \"Reversed\"))
  => [2 1]
  It's just like a comment... but you can turn it into (cs) so it will print the comment followed by the data sctructure passed in. You can also use (showc) to 'switch on' all the (c) functions in a given scope"
  data)

(defn c-print [f data s]
  "Use (showc) to add it as a hook on the (c) function, to 'switch it on' and make it behave like (cs) as long as the hook is added."
  (println s)
  (cprint data)
  data)

(defn cs [data s]
  "A debugging tool for ->> thread chains. Takes data in, returns the same data out, and prints your comment followed by the data. Can be switched off by turning it into (c)"
  (c-print c data s))


;;;; (cl) and (cls)
; TODO: for lazy sequences (for instance after a (map) in a thread chain)

(defn cl [data s]
  "Same as (c), but for lazy data. See the doc for (cls)."
  data)

(defn cl-print [f data s]
  "Same as (c-print) but for lazy data. See the doc for (cls)"
  (let [d (doall data)]
    (println s)
    (cprint d)
    d))

(defn cls [data s]
  "Same as (cs) but for lazy data. For instance when I do
  (-> some-data (map some-fn) (cs \"some comment\"))
  the (map) will return a lazy list, which gets evaluated only when the list is printed. This makes the text \"some comment\" be printed before (some-fn) gets evaluated, and the data be printed after it's evaluated. As a consequence, if (some-fn) prints anything, this text will end up between \"some comment\" and the relevant data.

  (cls) solves this problem by evaluating the lazy list before doing anything else. This is good for printing purpose, but it means that you are now manipulating a list instead of a lazy list - which might not be what you want."
  (cl-print cl data s))


;;;; (cc) and (ccs)

(defn cc [s data]
  "Same as (c) but for ->> thread chains."
  data)

(defn cc-print [f s data]
  "Same as (c-print) but for (cc)"
  (println s)
  (cprint data)
  data)

(defn ccs [s data]
  "Same as (cs)but for -> thread chains"
  (cc-print cc s data))

;;;; (ccl) and (ccls)
; TODO: for lazy sequences (for instance after a (map) in a thread chain)

(defn ccl [s data]
  "Same as (cl) but for (cc) rather than (c)"
  data)

(defn ccl-print [f s data]
  "Same as (cl-print) but for (cc) rather than (c)"
  (let [d (doall data)]
    (println s)
    (cprint d)
    d))

(defn ccls [s data]
  "Same as (cls) but for (cc) rather than (c)"
    (ccl-print ccl s data))


;;;; (showc) and (no-showc)

(defmacro showc [& forms]
   "Makes (c) behave like (cs) and (cc) like (ccs): they now print the comment and data.
   But only in the scope of (showc).
   You can use (no-showc) to cancel this behavior in a sub scope"
   (use 'robert.hooke)
   `(robert.hooke/with-scope
     (robert.hooke/add-hook #'c #'c-print)
     (robert.hooke/add-hook #'cc #'cc-print)
     (robert.hooke/add-hook #'cl #'cl-print)
     (robert.hooke/add-hook #'ccl #'ccl-print)
     ; Just the same forms as they were
     ~@forms
))

(defmacro no-showc [& forms]
  "In the given scope, don't let (c) behave like (cs), nor (cc) behave like (ccs)"
  (use 'robert.hooke)
  (robert.hooke/with-hooks-disabled c
    (robert.hooke/with-hooks-disabled cc
      (robert.hooke/with-hooks-disabled cl
        (robert.hooke/with-hooks-disabled ccl
          ; Just the same forms as they were
          ~@forms
       ))
)))
