(ns mascip.debug
  (:require [clojure.string :as s]
            [puget.printer  :refer :all]
            [robert.hooke   :refer :all]
  ))

"Acknowledgement:
 - The (c) function is inspired by Micheal Feather's c function for Ruby, in his article \"Literate Chains for Functional Programming\"
 - (dbg) was written by John Lawrence Aspden and I found it on Stackoverflow: http://stackoverflow.com/questions/2352020/debugging-in-clojure"

(defmacro dbg
  "Try this:
  (println (+ (* 2 3) (dbg (* 8 9))))
  => dbg: (* 8 9) = 72
  "
  [x]
  `(let [x# ~x] (println "dbg:" '~x "=>" x#) x#))

(defmacro pdbg [x]`(let [x# ~x] (do (println "--")(cprint '~x)(println "->")(cprint x#) (println "--") x#))) 

(defn c [data s]
  "A comment that helps you debug your -> thread chains. Is takes data in and passes the same data out. For example:
  (-> [1 2] (c \"The vector\") reverse (c \"Reversed\"))
  => [2 1]
  It's just like a comment... but you can turn it into (cs) so it will print the comment followed by the data sctructure passed in. You can also use (showc) to 'switch on' all the (c) functions in a given scope"
  data)

(defn c-print [f data s]
  ; FIXME: update doc
  "Add it as a hook on the (c) function, to 'switch it on' and make it behave like (cs) as long as the hook is added.
  For this, you have to wrap the code where you want to 'switch on' (c) into
  (with-scope (add-hook #'c #'c-print) ...your code here... ).
  For this you need to declare [robert.hook :refer :all] as a dependency...but I will soon create a macro so the syntax is simpler and you don't need to declare anything extra."
  (println s)
  (cprint data)
  data
)

(defn cs [data s]
  "A debugging tool for ->> thread chains. Takes data in, returns the same data out, and prints your comment followed by the data. Can be switched off by turning it into (c)"
  (c-print c data s))

(defn cc [s data]
  "Same as (c) but for ->> thread chains."
  (c data s))

(defn cc-print [f s data]
  "Same as (c-print) but for (cc)"
  (c-print f data s))

(defn ccs [s data]
  "Same as (cs)but for -> thread chains"
  (cc-print cc s data))

(defmacro showc [& forms]
   "Makes (c) behave like (cs) and (cc) like (ccs): they now print the comment and data.
   But only in the scope of (showc).
   You can use (no-showc) to cancel this behavior in a sub scope"
   (use 'robert.hooke)
  `(robert.hooke/with-scope
     (robert.hooke/add-hook #'c #'c-print)
     (robert.hooke/add-hook #'cc #'cc-print)
     ; Just the same forms as they were
     ~@forms
))

(defmacro no-showc [& forms]
  "In the given scope, don't let (c) behave like (cs), nor (cc) behave like (ccs)"
  (use 'robert.hooke)
  (robert.hooke/with-hooks-disabled c
    (robert.hooke/with-hooks-disabled cc
       ; Just the same forms as they were
       ~@forms
)))
