(ns structural-typing.assist.type-repo
  "The `TypeRepo` structure and its functions."
  (:use structural-typing.clojure.core)
  (:require [structural-typing.guts.type-descriptions :as type-descriptions]
            [structural-typing.guts.compile.compile :as compile]
            [structural-typing.assist.defaults :as defaults]))

(def valid-type-signifier?
  "A valid type signifier is either a keyword or a string."
  (some-fn keyword? string?))

(defprotocol TypeRepoLike
  (canonicalize [type-repo condensed-type-descriptions])
  (hold-type [type-repo type-signifier type-descriptions])
  (get-compiled-type [type-repo type-signifier])
  (replace-success-handler [type-repo handler]
    "For this `type-repo`, handle candidates that typecheck successfully by 
     passing them to `handler` as the last step in [[built-like]].
     Thus, `built-like` will return the handler's result.")
  (replace-error-handler [type-repo handler]
    "For this `type-repo`, pass [[oopsies]] generated by type failures to 
     `handler` as the last step in [[built-like]]. Thus, `built-like` will
     return the handler's result.")
  (the-success-handler [type-repo])
  (the-error-handler [type-repo]))

(defrecord TypeRepo [success-handler error-handler]
  TypeRepoLike
  (canonicalize [type-repo condensed-type-descriptions]
    (type-descriptions/canonicalize condensed-type-descriptions
                                    (:canonicalized-type-descriptions type-repo)))
    
  (hold-type [type-repo type-signifier condensed-type-descriptions]
    (let [canonicalized (canonicalize type-repo condensed-type-descriptions)
          compiled (compile/compile-type canonicalized)]
      (-> type-repo 
          (assoc-in [:original-type-descriptions type-signifier] condensed-type-descriptions)
          (assoc-in [:canonicalized-type-descriptions type-signifier] canonicalized)
          (assoc-in [:compiled-types type-signifier] compiled))))

  (get-compiled-type [type-repo type-signifier]
    (or (get-in type-repo [:compiled-types type-signifier])
        (boom! "There is no type `%s`" type-signifier)))

  (replace-error-handler [type-repo f]
    (assoc type-repo :error-handler f))
  
  (replace-success-handler [type-repo f]
    (assoc type-repo :success-handler f))
  
  (the-error-handler [type-repo] (:error-handler type-repo))
  (the-success-handler [type-repo] (:success-handler type-repo)))

(defmethod clojure.core/print-method TypeRepo [o, ^java.io.Writer w]
  (.write w "#TypeRepo[")
  (.write w (->> o :original-type-descriptions keys (str-join ", ")))
  (.write w "]"))
          

(def empty-type-repo
  "A type repo that contains no types and uses the default success and error handlers."
  (->TypeRepo defaults/default-success-handler defaults/default-error-handler))

(defn origin
  "Returns the original description of the `type-signifier` (a sequence of vectors and maps)"
  [type-repo type-signifier]
  (get-in type-repo [:original-type-descriptions type-signifier]))

(defn description
  "Returns the canonical (expanded) description of the `type-signifier`."
  [type-repo type-signifier]
  (get-in type-repo [:canonicalized-type-descriptions type-signifier]))
