(ns manenko.clj-jira.request
  "Functions that generate ring-style request maps for Jira REST API."
  (:require [manenko.clj-jira.middleware :as middleware]))


(defn get-current-user
  "Returns a ring-style request map to get information about a current user.

  **Arguments:**

  opts
  : An optional map that configures a Jira query and supports the following
    keys:
    * `:expand` - A sequence of fields to include to the response.  Supports
                  the following values: `:group` (returns all groups, including
                  nested groups, the user belongs to) and `:applicationRoles`
                  (returns the application roles the user is assigned to).

  The function creates a request that uses [Get current user] Atlassian Jira API.

  [Get current user]: https://developer.atlassian.com/cloud/jira/platform/rest/v3/#api-rest-api-3-myself-get"
  ([]
   (get-current-user {}))
  ([opts]
   {::middleware/api-resource "/myself"
    :method                   :get
    :query-params             (select-keys opts [:expand])}))


(defn get-projects
  "Returns a ring-style request map to get projects visible to a current user.

  **Arguments:**
  
  opts
  : An optional map that configures a Jira query and supports the following
    keys:
    * `:startAt`    - A page offset.  Integer.  Default is `0`.
    * `:maxResults` - A page size.  Integer.  Default is `0`.  Maximum is `0`.
    * `:orderBy`    - Orders results by a field.  The following values
                      are supported: `:category`, `:key`, `:name`, `:owner`.
                      Default is `:key`, which sorts projects alpabetically
                      by project key.  You can prepend value with `+` or `-`
                      to specify a sort direction, e.g. `:+owner`,
                      `:-category`, etc.
    * `:query`      - Searches for projects which `key` or `name` matches
                      the given string.
    * `:typeKey`    - Orders results by the project type.  A sequence of the
                      following values: `:business`, `:service_desk`,
                      `:software`.
    * `:categoryId` - An identifier of the project's category.  Integer.
    * `:searchBy`   - Default is `[:key, :name]`.
    * `:action`     - Filters results by projects for which user can view,
                      browse, or edit the project.  Supports the following
                      values: `:view`, `:browse`, `:edit`.  Default is `view`.
    * `:expand`     - A sequence of fields to include to the response.  Supports
                      the following values: `:description`, `:projectKeys`,
                      `:lead`, `:issueTypes`, `:url`.

  The function creates a request that uses [Get projects paginated] Atlassian Jira API.

  [Get projects paginated]: https://developer.atlassian.com/cloud/jira/platform/rest/v3/#api-rest-api-3-project-search-get"
  ([opts]
   {::middleware/api-resource "/project/search"
    :method                   :get
    :query-params             (select-keys opts
                                           [:action
                                            :categoryId
                                            :expand
                                            :maxResults
                                            :orderBy
                                            :query
                                            :searchBy
                                            :startAt
                                            :typeKey])})
  ([]
   (get-projects {})))
