(ns metaheuristics.annealing)

(defprotocol AnnealingProblem
  "A protocol that specifies the inputs to the annealing
  function."
  (temperature [this iteration] "Returns the temperature that we reach for the input iteration. Iteration increases by one
  for each 'time unit'.")
  (successor [this current-solution] "Returns a valid successor solution that can be reached from the current solution")
  (cost [this solution] "Returns the cost of the provided solution.")
  (accept [this current-cost candidate-cost current-temperature]
    "Returns a truthy value if we want to move to candidate-solution, and false if we want to stay."))

(defn anneal-seq
  "Returns a lazy sequence which iterates over solutions until we've 'cooled'. The starting state will be the
  first item returned in the sequence. The last item returned is the first state for which temp is less than
  or equal to 0."
  ([problem initial-state] (anneal-seq problem initial-state 0))
  ([problem current-solution iteration]
   (let [current-temp (temperature problem iteration)]
     (if (<= current-temp 0)
       (list current-solution)
       (cons current-solution
             (lazy-seq (let [next-candidate (successor problem current-solution)
                             current-cost (cost problem current-solution)
                             candidate-cost (cost problem next-candidate)
                             next-solution (if (< candidate-cost current-cost)
                                             ; automatically take a lower-cost state
                                             next-candidate
                                             ; shake
                                             (if (accept problem current-cost candidate-cost current-temp)
                                               next-candidate
                                               current-solution))]
                         (anneal-seq problem next-solution (inc iteration)))))))))

(defn anneal
  "Attempts to find an optimal solution to the provided AnnealingProblem."
  ([problem initial-state] (anneal problem initial-state 0))
  ([problem current-solution iteration]
   (last (anneal-seq problem current-solution iteration))))