(ns lupapiste-invoice-commons.states
  (:require [schema.core :as s]))


(def Actions (s/enum :add-to-transfer-batch :remove-from-transfer-batch))
(def RuleActions {(s/optional-key :next) [Actions]
                  (s/optional-key :previous) [Actions]})

(def StateName (s/enum "draft" "checked" "confirmed" "transferred" "billed"))
(def States [StateName])

(def RuleSet {:states States
              :rules {StateName {(s/optional-key :previous) StateName
                                 (s/optional-key :next) StateName
                                 :name StateName
                                 (s/optional-key :actions) RuleActions}}})

(def RuleSets {s/Keyword RuleSet})

(def draft "draft")
(def checked "checked")
(def confirmed "confirmed")
(def transferred "transferred")
(def billed "billed")

(def states [draft checked confirmed transferred billed])

(def backend-rule-set {:states [draft checked confirmed transferred billed]
                       :rules {draft {:next checked
                                      :name draft}
                               checked {:next confirmed
                                        :previous draft
                                        :actions {:previous [:remove-from-transfer-batch]}
                                        :name checked}
                               confirmed {:previous checked
                                          :actions {:next [:add-to-transfer-batch]}
                                          :name confirmed
                                          :next transferred}
                               transferred {:previous confirmed
                                            :name transferred
                                            :next billed}
                               billed {:previous transferred
                                       :name billed}}})

(def billing-ui-rule-set {:states [checked confirmed transferred billed]
                          :rules {checked {:next confirmed
                                           :previous draft
                                           :name checked}
                                  confirmed {:previous checked
                                             :name confirmed}
                                  transferred {:previous confirmed
                                               :name transferred
                                               :next billed}
                                  billed {:previous transferred
                                          :name billed}}})

(def admin-ui-rule-set {:states [draft checked confirmed transferred billed]
                        :rules {draft {:next checked
                                       :name draft}
                                checked {:previous draft
                                         :name checked}}})

(def default-rule-sets {:billing billing-ui-rule-set :admin admin-ui-rule-set :backend backend-rule-set})

(def MoveToStateResponse {:value s/Any :ok s/Bool :error s/Str :actions [Actions]})

(s/defn ^:always-validate  get-state-by-name :- StateName
  ([state-name :- StateName
    rule-set-key :- s/Keyword
    rule-sets :- RuleSets]
   (let [rule-set (rule-set-key rule-sets)
         states (:states rule-set)]
     (some (fn [state]
             (if (= state-name state) state)) states)))
  ([state-name :- StateName
    rule-set-key :- s/Keyword]
   (get-state-by-name state-name rule-set-key default-rule-sets)))

(s/defn ^:always-validate next-state :- (s/maybe StateName)
  ([current-state :- StateName
    rule-set-key :- s/Keyword
    rule-sets :- RuleSets]
   (let [rules (get-in rule-sets [rule-set-key :rules])]
     (get-in rules [current-state :next])))
  ([current-state :- StateName
    rule-set-key :- s/Keyword]
   (next-state current-state rule-set-key default-rule-sets)))

(s/defn ^:always-validate previous-state :- (s/maybe StateName)
  ([current-state :- StateName
    rule-set-key :- s/Keyword
    rule-sets :- RuleSets]
   (let [rules (get-in rule-sets [rule-set-key :rules])]
     (get-in rules [current-state :previous])))

  ([current-state :- StateName
    rule-set-key :- s/Keyword]
   (previous-state current-state rule-set-key default-rule-sets)))

(s/defn ^:always-validate can-move-to? :- s/Bool
  ([current-state :- StateName
    wanted-state :- StateName
    transition-key :- (s/enum :next :previous)
    rule-set-key :- s/Keyword
    rule-sets :- RuleSets]
   (let [rules (get-in rule-sets [rule-set-key :rules])
         allowed-state (get-in rules [current-state transition-key])]
     (= wanted-state allowed-state)))
  ([current-state :- StateName
    wanted-state :- StateName
    transition-key :- (s/enum :next :previous)
    rule-set-key :- s/Keyword]
   (can-move-to? current-state wanted-state transition-key rule-set-key default-rule-sets)))

(s/defn ^:always-validate move-to-state :- MoveToStateResponse
  ([path-to-update :- [s/Keyword]
    target :- s/Any
    wanted-state :- StateName
    transition-key :- (s/enum :next :previous)
    rule-set-key :- s/Keyword
    rule-sets :- RuleSets]
   (let [current-state (get-in target path-to-update)
         wanted-state-actions (get-in rule-sets [rule-set-key :rules wanted-state :actions transition-key])
         can-move? (can-move-to? current-state
                                 wanted-state
                                 transition-key
                                 rule-set-key
                                 rule-sets)]
     (if can-move?
       {:ok true
        :value (assoc-in target path-to-update wanted-state)
        :error ""
        :actions (or wanted-state-actions wanted-state-actions [])}
       {:ok false :value nil :error (str "Cannot move from " current-state " to " wanted-state) :actions []})))
  ([path-to-update :- [s/Keyword]
    target :- s/Any
    wanted-state :- StateName
    transition-key :- (s/enum :next :previous)
    rule-set-key :- s/Keyword]
   (move-to-state path-to-update target wanted-state transition-key rule-set-key default-rule-sets)))

(defn state-change-direction
  "Returns direction of state change.
  Parameters:
    `current-state`
    `to-state`
    `rule-set-key`
    `rule-sets` Optional, if not given we use default rulesets

  Returns :next :previous :same or nil (when transition is not valid for some reason)
  "
  ([current-state to-state rule-set-key rule-sets]
   (let [ruleset (rule-set-key rule-sets)
         rule-entry (get (:rules ruleset) current-state)
         previous-state (:previous rule-entry)
         next-state (:next rule-entry)]
     (cond
       (= current-state to-state) :same
       (= to-state previous-state) :previous
       (= to-state next-state) :next
       :else nil)))
  ([current-state to-state rule-set-key]
   (state-change-direction current-state to-state rule-set-key default-rule-sets)))
