(ns toshokan.gol
  (:require [reagent.core :as r]
            [toshokan.data :as data :refer [conn]]
            [cljs.core.async :refer [timeout <! close! chan put!]])
  (:require-macros [cljs.core.async.macros :refer [go-loop]]))


;; initial application state
(defonce selected-seed (r/atom "glider"))
(defonce sim-state (r/atom (data/get-seed @selected-seed)))
(defonce running (r/atom nil))
(defonce paused (r/atom false))
(defonce control-chan (chan))

;; cellular automaton functions
(defn neighbors
  "determine neighbors on a torus"
  [[x y] x-dim y-dim]
  (for [dx [-1 0 1]
        dy [-1 0 1]
        :when (not= [dx dy] [0 0])]
    [(mod (+ x dx) x-dim) (mod (+ y dy) y-dim)]))

(defn time-step
  "iterate one time step in the cellular automaton"
  [pop]
  (let [all-neighbors (mapcat #(neighbors % 50 50) pop)
        neighbor-count (frequencies all-neighbors)]
    (set (for [[cell count] neighbor-count
               :when (or (= count 3)
                         (and (= count 2)
                              (pop cell)))]
           cell))))

(defn game-loop
  "go-loop to allow pausing/unpausing and killing of the cellular automaton"
  [control]
  (go-loop [action :run]
    (let [[v c] (alts! [control (timeout 250)])]
      (if (= c control)
        (recur v)
        (do (if (= action :run)
              (swap! sim-state time-step))
            (if (not= action :kill)
              (recur action)))))))

(defn start-gol
  "start the cellular automaton"
  []
  (reset! running (game-loop control-chan)))

(defn pause-gol
  "pause and unpause the cellular automaton"
  []
  (if (not @paused)
    (put! control-chan :pause)
    (put! control-chan :run))
  (swap! paused not)
  ;; return nil to avoid React Event Handler warning
  nil)

(defn reset-gol
  "reset the  cellular automaton"
  []
  (if @running (put! control-chan :kill))
  (if @running (close! @running))
  (reset! running nil)
  (reset! paused false)
  (reset! sim-state (data/get-seed @selected-seed))
  ;; return nil to avoid React Event Handler warning
  nil)

(defn update-gol
  [seed]
  (reset! selected-seed seed)
  (reset-gol))


;; HTML DOM layout functions using Hiccup-style templating
(defn title
  [title-string]
  (fn [title-string]
    [:h1 title-string]))

(defn torus
  "cellular automaton grid"
  [cell-color]
  [:svg {:width 500
         :height 500
         :viewBox "0 0 500 500"}
   (doall
    (for [[x y] @sim-state]
      ^{:key [x y]} 
      [:rect {:width 8
              :height 8
              :x (+ (* x 10) 1)
              :y (+ (* y 10) 1)
              :style {:fill cell-color}}]))])

(defn buttons
  "cellular automaton control buttons"
  []
  [:p
   [:button {:id "gol-start-button"
             :disabled @running
             :on-click start-gol}
    "start"]
   [:button {:id "gol-pause-button"
             :disabled (not @running)
             :on-click pause-gol}
    (if @paused "unpause" "pause")]
   [:button {:id "gol-reset-button"
             :disabled (not @running)
             :on-click reset-gol}
    "reset"]])

(defn seed-list
  "list of gol seeds"
  []
  (fn []
    [:div {:id "seeds"}
     [:select {:id "seed-select"
               :value @selected-seed
               :on-change #(update-gol (-> % .-target .-value))}
      (doall
       (for [[e n] (data/get-seed-names)]
         ^{:key [e n]} [:option {:value n} n]))]]))

(defn check-input
  "checks input value against non-empty and a validation predicate
   and returns appropriate message"
  [val pred]
  (if (or (nil? val)
          (= "" val))
    "field cannot be empty"
    (if (pred val)
      "field is valid"
      "field is not valid")))

(defn input-elem
  "an input element template"
  [id value in-focus]
  [:input {:id id
           :name id
           :class "input-elem"
           :type id
           :required ""
           :value @value
           :on-change #(reset! value (-> % .-target .-value))
           :on-focus #(swap! in-focus not)
           :on-blur #(swap! in-focus not)}])

(defn input-with-msg
  "wraps and input element with a label and validation message"
  [label-val name elem-arg elem-valid req]
  (let [input-focus (r/atom false)] 
    (fn []
      [:div
       [:label label-val]
       [input-elem name elem-arg input-focus]
       [:div (check-input @elem-arg elem-valid)]])))

(defn seed-name-elem
  "creates an input element for a seed name"
  [seed-name-atom]
  (input-with-msg "Seed Name:"
                  "seed-name"
                  seed-name-atom
                  data/valid-seed-name?
                  true))

(defn seed-pattern-elem
  "creates an input element for a seed pattern"
  [seed-name-atom]
  (input-with-msg "Seed Pattern:"
                  "seed-pattern"
                  seed-name-atom
                  data/valid-seed-pattern?
                  true))

(defn add-seed
  "a pseudo-form for adding a new seed"
  []
  (let [seed-name (r/atom "")
        seed-pattern (r/atom "")]
    (fn []
      [:div {:class "gol-seed-wrapper"}
       [:h3 "Add a Seed"]
       [seed-name-elem seed-name]
       [seed-pattern-elem seed-pattern]
       [:button {:id "gol-insert-seed"
                 :disabled (or (not (data/valid-seed-name? @seed-name))
                               (not (data/valid-seed-pattern? @seed-pattern)))
                 :on-click #(do (data/insert-seed @seed-name  @seed-pattern)
                                (reset! seed-name "")
                                (reset! seed-pattern ""))}
        "insert"]])))
