(ns figwheel-testbook.reporting.report
  (:require
    [reagent.core :as r]
    [cljs.test :as test]
    [taoensso.timbre :as log :include-macros true]))

(defonce results (r/atom {}))
(defonce current* (atom nil))

(defn set-current-test! [ns test]
  (reset! current* [ns test]))

(defn current-test-path [suffix]
  (into [] (conj @current* suffix)))


(defmethod test/report [:cljs.test/default :begin-test-ns] [_])
(defmethod test/report [:cljs.test/default :end-test-ns] [_])

(defmethod test/report [:cljs.test/default :begin-test-var] [{:keys [var] :as report}]
  (let [{:keys [ns name]} (meta var)
        ns (keyword ns)
        name (keyword name)]
    (log/info "start:" ns name)
    (set-current-test! ns name)
    (swap! results
           #(update-in % [ns name] merge {:start     (.getTime (js/Date.))
                                          :errors    []
                                          :fails     []
                                          :output    []
                                          :old-print *print-fn*}))))

(defmethod test/report [:cljs.test/default :end-test-var] [{:keys [var] :as report}]
  (let [{:keys [ns name]} (meta var)
        ns (keyword ns)
        name (keyword name)]
    (log/info "finish:" ns name)
    (swap! results
           #(update-in % [ns name] merge {:end (.getTime (js/Date.))}))))

(defn dbg [x]
  (log/error x)
  x)

(defmethod test/report [:cljs.test/default :error] [m]
  (swap! results
         #(update-in % (current-test-path :errors)
                     conj (dbg (assoc m :context (:testing-contexts (test/get-current-env)))))))

(defmethod test/report [:cljs.test/default :fail] [m]
  (swap! results
         #(update-in % (current-test-path :fails)
                     conj (assoc m :context (:testing-contexts (test/get-current-env))))))

(defmethod test/report [:cljs.test/default :summary] [_])

;; Summary
;; -------

(defn summary-n [content summary-n+1]
  (let [sub-statuses (map #(-> % summary-n+1 :status) (vals content))]
    {:status
              (cond
                (contains? (set sub-statuses) :running) :running
                (contains? (set sub-statuses) :failed) :failed
                :else :success)
     :running (->> sub-statuses (filter #(= % :running)) count)
     :success (->> sub-statuses (filter #(= % :success)) count)
     :failed  (->> sub-statuses (filter #(= % :failed)) count)}))

;; Test
(defn running? [{:keys [start end] :as test-result}]
  (and start (or (nil? end) (< end start))))

(defn success? [{:keys [errors fails] :as test-result}]
  (and (empty? errors) (empty? fails)))

(defn summary-2 [{:keys [start end] :as content}]
  {:status
   (cond (running? content) :running
         (success? content) :success
         :else :failed)
   :duration
   (if (and start end (not (running? content)))
     (- end start)
     nil)})

;; Per NS
(defn summary-1 [results]
  (summary-n results summary-2))

;; TOTAL
(defn summary-0 [results]
  (summary-n results summary-1))



