(ns budb.register
  (:require [budb.core :as core]
            [budb.helpers :as h]
            [konserve.core :as k]
            [konserve.memory :as mem]
            [taoensso.timbre :as log :include-macros true]
    #?(:clj
            [clojure.spec.alpha :as s]
       :cljs [cljs.spec :as s :include-macros true])
    #?(:clj
            [clojure.core.async :refer [go >! <!] :as async]
       :cljs [cljs.core.async :as async :refer [<! >!]])
    #?(:clj
            [superv.async :refer [S go-try go-loop-try <? >?]]
       :cljs [superv.async :refer [S go-try go-loop-try <? >?]]))
  #?(:cljs
     (:require-macros
       [cljs.core.async.macros :refer [go]])))

(def register-lww-crdt
  {:state   {:valid?  any?
             :initial [0 nil]}
   :updates {:set {:valid?     (fn [_ _] true)
                   :prepare    (fn [replica-id state version value]
                                 [(h/time-ordered-unique-id replica-id) value])
                   :appliable? (fn [_ _] true)
                   :do         (fn [[current-id current-value] [new-id new-value]]
                                 (if (> current-id new-id)
                                   [current-id current-value]
                                   [new-id new-value]))}}
   :queries {:get (fn [[id value]]
                    value)}})

;; Interface
(defprotocol CRDVRegister
  (set-value! [this x ack-on])
  (get-value [this]))

;; Register Last Writer Win
(defrecord CRDVRegisterLWW [crdv]
  core/CRDV
  (wal [this]
    (core/wal-stream crdv))
  (replica-id [this]
    (core/replay! crdv))
  (sync! [this store]
    (go (<! (core/remote-pull crdv store))
        (<! (core/remote-push crdv store))))
  CRDVRegister
  (set-value! [this x ack-on]
    (core/until-ack ack-on (core/update! crdv :set x)))
  (get-value [this]
    (core/query? crdv :get)))

(defn make-lww [store & {:keys [replica-id]}]
  (go
    (map->CRDVRegisterLWW
      {:crdv (<! (-> register-lww-crdt
                     (core/make! :store (or store :mem) :replica-id replica-id)
                     <! core/start! core/replay!))})))

