(ns tango.ui.interactive
  (:require [reagent.impl.component :as r-component]
            [promesa.core :as p]
            [reagent.core :as r]
            [tango.integration.interpreter :as int]
            [tango.ui.elements :as ui]
            [orbit.evaluation :as eval]))

(defonce orig-wrap-render r-component/wrap-render)

(def ^:private wrappings (atom ()))
(defn- wrap-render-into-tries [errors]
  (swap! wrappings conj ::wrapped)
  (set! r-component/wrap-render (fn [c compiler]
                                  (try
                                    (orig-wrap-render c compiler val)
                                    (catch :default e
                                      (swap! errors conj [c e])
                                      (r/create-element "div"))))))

(defn- check-errors [hiccup]
  (try
    (let [errors (atom [])]
      (wrap-render-into-tries errors)
      @errors)
    (catch :default e
      [e])
    (finally
      (p/do!
       (p/delay 500)
       (when (empty? (swap! wrappings rest))
         (set! r-component/wrap-render orig-wrap-render))))))

(defn error-boundary [comp state]
  (let [editor-state (:editor-state state)
        renderer (-> @editor-state :editor/features :result-for-renderer)
        error (r/atom nil)]
    (r/create-class
     {:component-did-catch (fn [this e info])
      :get-derived-state-from-error (fn [e]
                                      (reset! error e)
                                      (def e e)
                                      #js {})
      :reagent-render (fn [comp]
                        (if-let [error @error]
                          (let [old-result @(:eval-result state)]
                            [error-boundary [:div.error.rows
                                             [:div.title "Something went wrong."]
                                             [:div.space]
                                             [renderer
                                              (assoc old-result :result error)
                                              editor-state]]
                             state])
                          comp))})))

(defn- eval-to-dom! [state result result-state outer]
  (let [evaluator (-> state :editor-state deref :editor/features :interpreter/evaluator)]
    (-> evaluator
        (eval/evaluate (-> result :html pr-str)
                       {:plain true
                        :options {:bindings {'?state result-state
                                             'eql (-> state
                                                      :editor-state
                                                      deref
                                                      :editor/features
                                                      :eql)}}})
        (p/then ui/dom)
        (p/catch #(ui/dom [ui/WithClass ["error"] (.-message ^js %)]))
        (p/then #(do
                   ; (prn :AND? % (.-innerHTML %))
                   (. outer replaceChildren %))))))

(defn interactive [result state]
  (let [outer (ui/dom [:div [:div "..."]])
        result-state (-> result :state atom)]
    (eval-to-dom! state result result-state outer)
    (add-watch result-state ::interactive (fn [_ _ _ new-val]
                                            (eval-to-dom! state result result-state outer)))
    outer))
