(ns tango.ui.console
  (:require [reagent.core :as r]
            [promesa.core :as p]
            ["ansi_up" :default Ansi]
            [orbit.shadow.evaluator :as shadow-eval]
            [tango.ui.elements :as ui]
            [clojure.string :as str]))

(defn- create-div [ & class-names]
  (let [div (js/document.createElement "div")
        classes (.-classList div)]
    (doseq [c class-names] (.add classes c))
    div))

(defn view []
  (let [div (create-div "tango" "console")
        tab-div (create-div "details")
        content-div (create-div "content-part")]
    (set! (.-ansi div) (new Ansi))
    (set! (.-tab div) tab-div)
    (set! (.-content div) content-div)
    (doto div
          (.appendChild tab-div)
          (.appendChild content-div))))

(defn all-scrolled? [console]
  (let [content (.-content console)]
    (>= (.-scrollTop content)
        (- (.-scrollHeight content)
           (.-clientHeight content)))))

(defn scroll-to-end! [console]
  (let [content (.-content console)]
    (set! (.-scrollTop content) (.-scrollHeight content))))

(defn- create-content-span [^js console text]
  (let [ansi (.-ansi console)
        html (. ansi ansi_to_html text)
        html (str/replace html #"\n" "<br />")
        content-span (js/document.createElement "span")]
    (set! (.-innerHTML content-span) html)
    content-span))

(defn- create-textual-elements [^js console stream text]
  (let [cell (create-div "cell")
        gutter (create-div "gutter" (name stream))
        content (create-div "content" (name stream))
        content-span (create-content-span console text)
        icon-container (create-div "tango" "icon-container")]
    (. cell appendChild gutter)
    (. gutter appendChild icon-container)
    (. icon-container appendChild (create-div "icon"
                                              (if (= stream :out)
                                                "icon-quote"
                                                "icon-alert")))
    (. cell appendChild content)
    (. content appendChild content-span)
    (. (.-content console) appendChild cell)))

(defn- remove-all-pending [^js console]
  (mapv (fn [cell] (.removeChild (.-content console) (.-parentNode cell)))
        (. console querySelectorAll ".content.pending")))

(defn- append-text [^js console stream text]
  (let [scrolled-to-bottom? (all-scrolled? console)
        old-pendings (remove-all-pending console)
        last-gutter-classes (into #{}
                                  (some-> console
                                          (.querySelector ".cell:last-child .gutter")
                                          .-classList))]
    (if (contains? last-gutter-classes (name stream))
      (let [last-content (.querySelector console ".cell:last-child .content")]
        (.appendChild last-content (create-content-span console text)))
      (create-textual-elements console stream text))
    (doseq [cell old-pendings]
      (.appendChild (.-content console) cell))
    (when scrolled-to-bottom?
      (scroll-to-end! console))))

(defn append [console element icons]
  (let [scrolled-to-bottom? (all-scrolled? console)
        cell (create-div "cell")
        gutter (create-div "gutter")]
    (. cell appendChild gutter)
    (. gutter appendChild (apply create-div "icon" icons))
    (. cell appendChild element)
    (. (.-content console) appendChild cell)
    (when scrolled-to-bottom?
      ;; Rendering stuff is async, and we don't really know when it'll end
      ;; so we first scroll fast, and if it delays a little, we scroll again
      (js/setTimeout #(scroll-to-end! console) 60)
      (js/setTimeout #(scroll-to-end! console) 200)
      (js/setTimeout #(scroll-to-end! console) 500))))

(defn stdout [console text] (append-text console :out text))
(defn stderr [console text] (append-text console :err text))
(defn clear [^js console]
  (set! (.. console -content -innerHTML ) ""))

(defn- create-shadow-tab! [^js console]
  (when-not (.-shadowTab console)
    (set! (.-shadowTab console)
      (doto (create-div "shadow-cljs")
            (.. -classList (add "clients"))))
    (.appendChild (.-tab console) (.-shadowTab console))
    (.appendChild (.-shadowTab console)
                  (doto (create-div "title") (-> .-innerText (set! "Shadow-CLJS"))))
    (.appendChild (.-shadowTab console) (create-div "build-ids"))
    (.appendChild (.-shadowTab console) (create-div "individual-clients")))

  (.-shadowTab console))

(defn update-clients [^js console connection shadow-clients]
  (let [div (create-shadow-tab! console)
        all-clients (.querySelector div "div.individual-clients")
        select (js/document.createElement "select")
        curr-client-id (-> @connection :repl/evaluator shadow-eval/current-client-id)]
    (set! (.-onchange select)
      (fn [evt]
        (-> @connection :repl/evaluator
            (shadow-eval/change-client-id! (-> evt .-target .-value js/parseInt)))))
    (set! (.-innerHTML all-clients) "")
    (doseq [[client-id {:keys [client-info]}] shadow-clients
            :let [shadow-s (str "Build #" client-id " "
                                (:desc client-info)
                                " (" (-> client-info :build-id name) ")")
                  opt (js/document.createElement "option")]]
      (set! (.-innerText opt) shadow-s)
      (set! (.-value opt) client-id)
      (.appendChild select opt)
      (when (= client-id curr-client-id) (. opt setAttribute "selected" "selected")))

    (.appendChild all-clients (doto (js/document.createElement "label")
                                    (-> .-innerText (set! "Client to eval "))
                                    (.appendChild select)))
    (when-not (.querySelector select "option[selected]")
      (-> @connection :repl/evaluator
          (shadow-eval/change-client-id! (-> shadow-clients keys first))))))

(defn update-build-status [^js console compile-info]
  (let [div (create-shadow-tab! console)
        builds (.querySelector div "div.build-ids")
        build-id (-> compile-info :build-id name)
        build-id-div (if-let [div (.querySelector builds (str "." build-id))]
                       div
                       (.appendChild builds
                                     (create-div build-id)))]
    (set! (.-innerHTML build-id-div)
      (str
       build-id " <span class='tango icon-container'>"
       (case (-> compile-info :build-status :status)
         :failed "<div class='error'><span class='icon stop'></span></div>"
         :completed "<span class='icon check'></span>"
         "<span class='icon loading'></span></span")))))

(defn all-parsed-results [^js console]
  (for [content (.querySelectorAll console "div.content")
        :let [parsed (.-parsed content)]
        :when parsed]
    parsed))
