(ns tango.commands-to-repl.pathom
  (:require [promesa.core :as p]
            [duck-repled.core :as duck]
            [duck-repled.repl-protocol :as duck-repl]
            [com.wsscode.pathom3.connect.operation :as connect]
            [com.wsscode.pathom3.connect.planner :as pcp]
            [com.wsscode.pathom3.connect.operation :as pco]
            [com.wsscode.pathom3.connect.runner :as runner]

            [clojure.string :as str]
            ["path" :as path]))

(defn- connect-viz [plan-cache]
  (fn [env]
    (-> env
        (pcp/with-plan-cache plan-cache)
        #_
        (cond-> js/goog.DEBUG
          (p.connector/connect-env "Chlorine")))))

(defn- gen-eql [resolvers]
  (let [cache (atom {})]
    (duck/gen-eql {:resolvers resolvers :plugin (connect-viz cache)})))

(defn- resolvers-from-state [editor-state]
  (p/let [{:keys [editor/callbacks]} @editor-state
          config-dir (:config/directory @editor-state)
          editor-data ((:editor-data callbacks))
          config ((:get-config callbacks))
          not-found :com.wsscode.pathom3.connect.operation/unknown-value
          is-config? (-> config-dir
                         (path/relative (:filename editor-data))
                         (str/starts-with? "..")
                         not)]
    {:editor/data (or editor-data not-found)
     :config/eval-as (if is-config? :clj (:eval-mode config))
     :config/project-paths (vec (:project-paths config))
     :repl/evaluator (if is-config?
                       (-> @editor-state :editor/features :interpreter/evaluator)
                       (:repl/evaluator @editor-state))
     :config/repl-kind (-> @editor-state :repl/info :kind)}))

(def ^:private doc-part
  '(when-let [text (:doc @?state)]
     [:<>
      [:div.space]
      (if (:markdown? @?state)
        [ui/Markdown text]
        [:div.pre.text text])]))

(def ^:private spec-part
  '(when-let [spec (:spec @?state)]
     [:<>
      [:div.space]
      [:div.pre
       (cond-> "Spec:\n"
               (:args spec) (str "  args: " (pr-str (:args spec)) "\n")
               (:ret spec) (str "  ret: " (pr-str (:ret spec)) "\n")
               (:fn spec) (str "  fn: " (pr-str (:fn spec))))]]))

(def ^:private markdown-check
  '(when (:doc @?state)
     [:<>
      [:div.space]
      [:label [:input {:type :checkbox
                       :checked (:markdown? @?state)
                       :on-click (fn [e]
                                   (swap! ?state update :markdown? not))}]
       " Use markdown"]]))

(def ^:private var-contents
  '(let [get-source (fn [evt]
                      (.preventDefault evt)
                      (.stopPropagation evt)
                      (p/let [info (eql
                                    [{:editor/contents
                                      [{:text/current-var [:definition/source]}]}])]
                        (swap! ?state
                               assoc
                               :var-value
                               (-> info
                                   :editor/contents
                                   :text/current-var
                                   :definition/source
                                   :text/contents))))]
     (if (empty? (:arglists @?state))
       [:div.rows
        [:div.space]
        (if (contains? @?state :var-value)
          [ui/Markdown (str "```\n" (:var-value @?state) "\n```")]
          [:div [:a {:href "#"
                     :on-click get-source}
                 "Get contents of var"]])]
       [:div.rows
        [:div.space]
        (if (contains? @?state :var-value)
          [ui/Markdown (str "```\n" (:var-value @?state) "\n```")]
          [:div [:a {:href "#"
                     :on-click get-source}
                 "Get source"]])])))

(defn- improved-doc-for-var [{:var/keys [fqn meta spec]}]
  {:render/doc
   {:html [:div.rows
           '[ui/Cols
             (cond-> [:<>]
               (:macro @?state) (conj [:i "^:macro "])
               (:private @?state) (conj [:i "^:private "]))
             [ui/Title (-> @?state :fqn str)]]
           '(when-let [args (seq (or (:method-params @?state)
                                     (:arglists @?state)))]
              (map (fn [a] [:li {:key a} (pr-str a)]) args))

           ; '(cond-> [:div.cols]
           ;          (:macro @?state) (conj [:i "macro"])
           ;          (:private @?state) (conj [:i "private"]))
           doc-part
           spec-part
           markdown-check
           var-contents]
    :state (-> meta
               (dissoc :ns)
               (assoc :markdown? true :fqn fqn :spec spec))
    :fns {:get-contents '(fn [_ state value]
                           (assoc state :var-value value))}}})

(defn generate-eql-from-state! [state]
  (let [resolver #(resolvers-from-state state)
        resolvers (duck/add-resolver {:inputs []
                                      :outputs [:editor/data
                                                :config/eval-as
                                                :config/project-paths
                                                :repl/evaluator
                                                :config/repl-kind]}
                                     resolver)
        resolvers (duck/add-resolver resolvers
                                     {:inputs [:var/fqn :var/meta (connect/? :var/spec)]
                                      :outputs [:render/doc]}
                                     improved-doc-for-var)
        generated-eql (gen-eql resolvers)]
    (swap! state assoc-in [:pathom :original-resolvers] resolvers)
    (swap! state assoc-in [:pathom :global-resolvers] resolvers)
    (swap! state assoc-in [:editor/features :eql] generated-eql)))

(defn reset-resolvers [state]
  (swap! state update :pathom
         (fn [pathom-config]
           (assoc pathom-config
                  :global-resolvers (:original-resolvers pathom-config)
                  :eql (gen-eql (:original-resolvers pathom-config))))))

(defn- change-resolvers [state-val config fun duck-function]
  (let [old (-> state-val :pathom :global-resolvers)
        new (duck-function old config fun)]
    (-> state-val
        (assoc-in [:pathom :global-resolvers] new)
        (assoc-in [:editor/features :eql] (gen-eql new)))))

(defn add-resolver [state config fun]
  (swap! state change-resolvers config fun duck/add-resolver))

(defn compose-resolver [state config fun]
  (swap! state change-resolvers config fun duck/compose-resolver))
