(ns tango.ui.interactive
  (:require [reagent.impl.component :as r-component]
            [promesa.core :as p]
            [reagent.core :as r]
            [tango.integration.interpreter :as int]
            [orbit.evaluation :as eval]))

(defonce orig-wrap-render r-component/wrap-render)

(def ^:private wrappings (atom ()))
(defn- wrap-render-into-tries [errors]
  (swap! wrappings conj ::wrapped)
  (set! r-component/wrap-render (fn [c compiler]
                                  (try
                                    (orig-wrap-render c compiler val)
                                    (catch :default e
                                      (swap! errors conj [c e])
                                      (r/create-element "div"))))))

(defn- check-errors [hiccup]
  (try
    (let [errors (atom [])]
      (wrap-render-into-tries errors)
      @errors)
    (catch :default e
      [e])
    (finally
      (p/do!
       (p/delay 500)
       (when (empty? (swap! wrappings rest))
         (set! r-component/wrap-render orig-wrap-render))))))

(defn error-boundary [comp state]
  (let [editor-state (:editor-state state)
        renderer (-> @editor-state :editor/features :result-for-renderer)
        error (r/atom nil)]
    (r/create-class
     {:component-did-catch (fn [this e info])
      :get-derived-state-from-error (fn [e]
                                      (reset! error e)
                                      (def e e)
                                      #js {})
      :reagent-render (fn [comp]
                        (if-let [error @error]
                          (let [old-result @(:eval-result state)]
                            [error-boundary [:div.error.rows
                                             [:div.title "Something went wrong."]
                                             [:div.space]
                                             [renderer
                                              (assoc old-result :result error)
                                              editor-state]]
                             state])
                          comp))})))

(defn interactive [state]
  (let [old-state state
        result @(:result state)
        state (-> result :state r/atom)
        evaluator (-> old-state :editor-state deref :editor/features :interpreter/evaluator)
        html (r/atom [:div.tango.icon-container [:div.icon.loading]])
        ;; INFINITE LOOP warning - don't try to use `r/with-let`
        res (fn [] [error-boundary @html old-state])]
    (-> evaluator
        (eval/evaluate (str "[(fn [] " (-> result :html pr-str) ")]")
                       {:plain true
                        :options {:bindings {'?state state
                                             'eql (-> old-state
                                                      :editor-state
                                                      deref
                                                      :editor/features
                                                      :eql)}}})
        (p/then #(reset! html %)))
    [res]))
