(ns tango.ui.edn
  (:require [reagent.core :as r]
            ["react" :as react]
            [orbit.serializer :as serializer]
            [promesa.core :as p]
            [reagent.ratom :as ratom]
            [clojure.string :as str]
            [cljs.js :as j]
            [cljs.repl :as repl]
            [tango.ui.interactive :as int]
            [tango.integration.repl-helpers :as helpers]
            [tango.ui.elements :refer [Children Rows Cols Whitespace WithClass
                                       Title open-close]]))

(declare as-html)

(defn- create-child-state [state & keys]
  (let [key-path (cons :child-state keys)]
    (-> (:state state)
        (swap! update-in key-path
               #(or % (r/atom (:child-state state {}))))
        (get-in key-path))))

(defn custom-cursor [ratom index]
  (let [result (:result @ratom)
        metadata (meta result)]
    (cond
      (or (:orbit.ui.reflect/info metadata) (:orbit.ui/lazy metadata))
      (ratom/make-reaction
       (fn []
         (if (= :meta index)
           (update @ratom :result meta)
           (update @ratom :result vary-meta dissoc
                   :orbit.ui.reflect/info :orbit.ui/lazy)))

       :on-set (fn [_ {:keys [result]}]
                 (swap! ratom update :result
                        #(if (= :meta index)
                           (with-meta % result)
                           (with-meta result (meta %))))))

      (serializer/tagged-literal-with-meta? result)
      (ratom/make-reaction
       (fn []
         (if (= index :tag)
           (update @ratom :result #(.-tag %))
           (update @ratom :result #(.-form %))))

       :on-set (fn [_ {:keys [result]}]
                 (swap! ratom update :result
                        #(if (= :tag index)
                           (serializer/tagged-literal-with-meta result (.-form %))
                           (serializer/tagged-literal-with-meta (.-tag %) result)))))

      (map? result)
      (let [[index what] index
            idx-map (-> result seq (nth index) first atom)]
        (ratom/make-reaction
         #(if (= what :key)
            (assoc @ratom :result @idx-map)
            (update @ratom :result get @idx-map))
         :on-set (fn [_ {:keys [result]}]
                   (if (= :key what)
                     (let [old-key @idx-map]
                       (reset! idx-map result)
                       (swap! ratom update :result
                              #(-> %
                                   (assoc result (get % old-key))
                                   (dissoc old-key))))
                     (swap! ratom assoc-in [:result @idx-map]
                            result)))))

      (set? result)
      (let [idx-map (atom (nth (seq result) index))]
        (ratom/make-reaction
         #(update @ratom :result get @idx-map)
         :on-set (fn [_ {:keys [result]}]
                   (let [old-val @idx-map]
                     (reset! idx-map result)
                     (swap! ratom update :result
                            #(-> %
                                 (disj old-val)
                                 (conj result)))))))

      (list? result)
      (ratom/make-reaction
       #(update @ratom :result nth index)
       :on-set (fn [_ {:keys [result]}]
                 (swap! ratom update :result
                        (fn [lst]
                          (let [[s e] (split-at index lst)]
                            (apply list (concat s (->> e rest (cons result)))))))))
      :else (ratom/make-reaction
             #(update @ratom :result get index)
             :on-set (fn [_ {:keys [result]}]
                       (swap! ratom assoc-in [:result index] result))))))

(defn- update-state [state index-or-atom merge-data child-state]
  (let [new-eval-result (if (satisfies? IDeref index-or-atom)
                          index-or-atom
                          (custom-cursor (:eval-result state) index-or-atom))]
    (-> state
        (merge merge-data)
        (assoc :state child-state)
        (assoc :eval-result new-eval-result))))

(defn- clj-stack-row [[class method file row] state]
  (let [eval-result (:eval-result state)
        clj-file? (re-find #"\.(clj.?|bb)$" (str file))
        file (if (re-find #"(NO.FILE|form\-init\d+)" file)
               (:filename @eval-result "NO-FILE")
               file)]
    [Cols  "  at "
     (if clj-file?
       [WithClass ["stack" "clojure"]
        [:<>
         (str (demunge (str class)) " (")
         [:a {:href "#" :on-click (fn [evt])}
          (str file ":" row)]
         ")"]]
       [WithClass ["stack" "non-clojure"]
        [Cols
         (str class)
         "."
         (str method)
         (str " (" file ":" row ")")]])]))

(defn- demunge-js-name [js-name]
  (if (str/starts-with? js-name "global.")
    js-name
    (-> js-name
        (str/replace #"\$" ".")
        (str/replace #"(.*)\.(.*)$" "$1/$2")
        demunge)))

(defn goto-stacktrace [e state file row col]
  (.preventDefault e)
  (.stopPropagation e)
  (p/let [editor-state @(:editor-state state)
          eql (-> editor-state :editor/features :eql)
          open (-> editor-state :editor/callbacks :open-editor)
          result (eql {:ex/filename file
                       :ex/row (js/parseInt row)
                       :ex/col (some-> col js/parseInt)}
                      [:definition/filename :definition/row :definition/col])]
    (open {:file-name (:definition/filename result)
           :line (:definition/row result)
           :column (:definition/col result)})))

(defn- generic-stacktrace-row [{:keys [what file row col]} state]
  [Cols
   "  at "
   [WithClass ["stack"]
    [:<>
     (when what (str what " ("))
     [:a {:href "#" :on-click #(goto-stacktrace % state file row col)}
      (str file ":" row (when col (str ":" col)))]
     (when what ")")]]])

(defn- generic-stacktrace [state]
  (into [Rows]
        (map #(generic-stacktrace-row % state)
             @(:result state))))

(defn- cljs-stack-row [stack-row state]
  (when (and (re-find #"^\s+at" stack-row)
             (not (re-find #"\<eval\>" stack-row)))
    (let [found (re-find #"at (.*) \((.*?):(\d+):?(\d+)?\)" stack-row)
          [_ what file row col] (if found
                                  found
                                  (concat [nil nil]
                                          (rest (re-find #"at (.*?):(\d+):?(\d+)?" stack-row))))]
      [Cols
       "  at "
       [WithClass ["stack" "js"]
        [:<>
         (if what (demunge-js-name what) "<unknown>")
         " ("
         [:a {:href "#" :on-click #(goto-stacktrace % state file row col)}
          (str file ":" row (when col (str ":" col)))]
         ")"]]])))

(defn- error-result [state]
  (let [result (:result state)
        root? (:root? state)
        {:keys [open? link]} (when root? (open-close (:state state) false))
        tag-part (-> @result .-tag pr-str)
        form-part (.-form @result)
        form-part (cond-> form-part
                    (serializer/tagged-literal-with-meta? form-part) .-form)
        eval-result (:eval-result state)
        child-result (dissoc form-part :via :trace :cause)
        form-ratom (r/atom (assoc @eval-result :result child-result))
        child-state (update-state state form-ratom {:root? root?}
                                  (create-child-state (assoc state :child-state {:open? false})))
        stack-row (if (-> form-part :trace string?)
                    cljs-stack-row
                    clj-stack-row)]
    [Rows
     [WithClass (when root? "title") (str "#error " (or (:cause form-part)
                                                        (-> form-part :via first :message)
                                                        (:message form-part)))]
     (when root?
       [:<>
        (when (seq child-result)
          [Children [as-html child-state]])

        (when-let [via (:via form-part)]
          (->> via
               (map (fn [data]
                      (let [data (if (serializer/tagged-literal-with-meta? data)
                                   (.-form data)
                                   data)]
                        [Cols (when-let [type (-> data :type)]
                                (str (pr-str type) ": "))
                              (-> data :message str)])))
               (into [Children [:div.space] "Via:"])))

        (when-let [stack (:trace form-part)]
          (->> (cond-> stack (string? stack) str/split-lines)
               (map (fn [row] [stack-row row state]))
               (into [Children [:div.space] "Stacktrace:"])))])]))

(defn- tagged [state]
  (if (-> @(:result state) .-tag (= 'error))
    (error-result state)
    (let [result (:result state)
          root? (:root? state)
          child-state (create-child-state (assoc state :child-state {:open? true}))
          {:keys [open? link]} (when root? (open-close (:state state) false))
          tag-part (-> @result .-tag pr-str)
          form-cursor (custom-cursor (:eval-result state) :form)
          parent-state (update-state state form-cursor {:root? false} child-state)]

      (cond
        (not root?) [Rows [Cols "#" tag-part " " [as-html parent-state]]]
        open? [Rows [Cols link "#" tag-part]
               [Children [as-html (update-state state
                                                form-cursor
                                                {:root? true}
                                                child-state)]]]
        :closed? [Cols link "#" tag-part " " [as-html parent-state]]))))

(defn leaf [state]
  [Rows (pr-str @(:result state))])

(defn raw-data [state]
  [Rows (-> state :result deref :data)])

(defn- collection [open state close]
  (let [children-states (->> state
                             :result
                             deref
                             (map-indexed (fn [i _]
                                            [(custom-cursor (:eval-result state) i)
                                             (create-child-state state i)])))
        children (->> children-states
                      (map (fn [[cursor child-state]]
                             [as-html
                              (update-state state
                                            cursor
                                            {:root? true}
                                            child-state)])))
        parent (->> children-states
                    (map (fn [[cursor child-state]]
                           [as-html (update-state state
                                                  cursor
                                                  {:root? false}
                                                  child-state)]))
                    (interpose Whitespace)
                    (into [:<>]))
        {:keys [open? link]} (open-close (:state state) false)]

    (cond
      (-> state :root? not) [Cols open parent close]
      open? [Rows [Cols link " " open parent close]
             (into [Children] children)]
      :closed? [Rows [Cols link " " open parent close]])))

(defn as-vector [state]
  (collection "[" state "]"))

(defn as-set [state]
  (collection "#{" state "}"))

(defn as-list [state]
  (collection "(" state ")"))

(def Comma (react/createElement "div" #js {:className "inline"} ", "))
(defn as-map [state]
  (let [ratom (:eval-result state)
        children-states (->> state
                             :result
                             deref
                             (map-indexed (fn [i _]
                                            [(custom-cursor ratom [i :key])
                                             (custom-cursor ratom [i :val])
                                             (create-child-state state i :key)
                                             (create-child-state state i :val)])))
        children (->> children-states
                      (map (fn [[k-cursor v-cursor k-state v-state]]
                             [:<>
                              (WithClass
                               ["map-key" "opened"]
                               [as-html
                                (update-state state
                                              k-cursor
                                              {:root? true}
                                              k-state)])
                              [as-html
                               (update-state state
                                             v-cursor
                                             {:root? true}
                                             v-state)]]))
                      (interpose Whitespace))
        parent (->> children-states
                    (map (fn [[k-cursor v-cursor k-state v-state]]
                           [:<>
                            (WithClass
                             ["map-key" "closed"]
                             [as-html (update-state state
                                                    k-cursor
                                                    {:root? false}
                                                    k-state)])
                            Whitespace
                            [as-html (update-state state
                                                   v-cursor
                                                   {:root? false}
                                                   v-state)]]))
                    (interpose Comma)
                    (into [:<>]))
        {:keys [open? link]} (open-close (:state state) false)]

    (cond
      (-> state :root? not) [Cols "{" parent "}"]
      open? [Rows [Cols link "{" parent "}"]
             (into [Children] children)]
      :closed? [Rows [Cols link "{" parent "}"]])))

(defn- find-right-seq [state key]
  (->> @state
       :result
       keys
       (map-indexed (fn [i k] [i (= k key)]))
       (filter peek)
       ffirst))

(defn- reflection-info-details [{:keys [eval-result] :as state}]
  (let [index (find-right-seq eval-result :orbit.ui.reflect/info)
        cursor (custom-cursor eval-result [index :val])
        new-state (update-state state cursor {} (:state state))]
    (->> @cursor
         :result
         (map-indexed (fn [parent-idx {:keys [title contents]}]
                        (let [curr-cursor (custom-cursor (:eval-result new-state)
                                                         parent-idx)
                              funs-cursor (custom-cursor curr-cursor
                                                         [(find-right-seq curr-cursor :contents)
                                                          :val])]
                          (when-not (map? contents)
                            (->> contents
                                 (map-indexed (fn [idx _]
                                                [as-html (update-state
                                                          new-state
                                                          (custom-cursor funs-cursor idx)
                                                          {:root? true}
                                                          (create-child-state new-state
                                                                              parent-idx
                                                                              idx))]))
                                 (into [Rows [:div.space] [Title title]]))))))
         (into [:<>]))))

(defn MoreInfo [callback]
  [:a.more-info {:href "#" :on-click (fn [^js e]
                                         (.preventDefault e)
                                         (.stopPropagation e)
                                         (callback))}
    ""])

(defn- reflect [{:keys [eval-result] :as state}]
  (let [reflection-info (custom-cursor eval-result :meta)
        without-meta (custom-cursor eval-result :data)
        ratom (:state state)
        reflection-info-open? (:reflection-info-open? @ratom)]

    [Cols [as-html (update-state state
                                 without-meta
                                 {:root? (and (:root? state)
                                              (not reflection-info-open?))}
                                 (:state state))]
     (when (:root? state)
       [:<>
        Whitespace
        (when (and (:root? state) (not reflection-info-open?) (not (:open? @ratom)))
          [MoreInfo #(swap! ratom assoc :reflection-info-open? true)])
        (when (and (:root? state) reflection-info-open? (not (:open? @ratom)))
          [Children
           [Cols
            [Title "More Info"]
            Whitespace
            [:a {:href "#" :on-click (fn [^js e]
                                       (.preventDefault e)
                                       (.stopPropagation e)
                                       (swap! ratom assoc :reflection-info-open? false))}
             "Hide"]]
           [reflection-info-details (update-state state
                                                  reflection-info
                                                  {}
                                                  (:state state))]])])]))

(defn- lazy [{:keys [eval-result] :as state}]
  (let [reflection-info (custom-cursor eval-result :meta)
        metadata (custom-cursor eval-result :meta)
        without-meta (custom-cursor eval-result :data)
        ratom (:state state)
        reflection-info-open? (:reflection-info-open? @ratom)]
    [Cols
     [as-html (update-state state
                            without-meta
                            {:root? (not reflection-info-open?)}
                            (:state state))]
     (when (:root? state)
       [:<>
        Whitespace
        [MoreInfo (fn []
                    (p/let [eql (-> state :editor-state deref :editor/features :eql)
                            result (eql {:v {:text/contents (-> @eval-result :result pr-str)}}
                                        [{:v [:repl/result]}])
                            new-meta (-> result :v :repl/result :result meta)]
                      (swap! ratom assoc :reflection-info-open? true)
                      (swap! metadata assoc :result new-meta)))]])]))

(defn- wrapped-error [state]
  (let [child (custom-cursor (:eval-result state) 0)
        new-state (update-state state child {} (:state state))]
    [WithClass ["error"]
     [as-html new-state]]))

(defn- text-with-stacktrace [state texts-or-traces]
  (into [:div.rows]
        (for [row texts-or-traces]
          (if (map? row)
            [:div.children
             [:div.cols
              "at "
              [:a {:href "#" :on-click #(goto-stacktrace % state (:file row)
                                                         (:line row) (:column row))}
               (str (:resource-name row (:file row))
                    ":" (:line row)
                    (when-let [col (:column row)]
                      (str ":" col)))]]]
            row))))

#_(nil 10)
(defn- shadow-error [error state]
  (let [traces (re-seq #" File: (.*?):(\d+):(\d+)" error)]
    (->> traces
         (map (fn [[_ file row col?]]
                (cond-> {:file file :line row}
                  col? (assoc :column col?))))
         (into [[Title "Errors found in compilation process"]
                [:div.space]
                [:div error]])
         (text-with-stacktrace state))))

(defn- shadow-warnings [warnings state]
  (let [all-warnings (for [warning warnings]
                       [[:div.error (:msg warning)]
                        (when-let [e (:source-excerpt warning)]
                          [:code
                           (->> e
                                :before
                                (map #(vector :div.block (if (seq %) % " ")))
                                (into [:<>]))
                           [:div (:line e)]
                           [:div (str (->> " "
                                           (repeat (-> warning :column dec))
                                           (apply str))
                                      "^")]])
                        warning
                        [:div.space]])]
    (->> all-warnings
         (mapcat identity)
         (into [[Title "Warnings"]
                [:div.space]])
         (text-with-stacktrace state))))

(defn- shadow-errors [state]
  (let [eval-result @(:eval-result state)
        result (:result eval-result)
        warnings (:orbit.shadow/warnings result)
        norm-warnings (map (fn [warning]
                             (if (:file warning)
                               warning
                               (-> warning
                                   (assoc :file (:editor/filename eval-result))
                                   (update :line + -1 (-> eval-result :editor/data :range ffirst))
                                   (update :column + (-> eval-result :editor/data :range first second)))))
                           warnings)]
    (cond
      warnings (shadow-warnings norm-warnings state)

      (-> result :orbit.shadow/error (= :no-runtime-selected))
      [Cols "No javascript runtime running right now"]
      :else (shadow-error (:orbit.shadow/errors result) state))))

(defn- as-html [state]
  (let [result (-> state :eval-result (r/cursor [:result]))
        new-state (assoc state :result result)
        metadata (meta @result)]
    (when-let [id (:orbit.patch/id metadata)]
      (swap! (:patches state) assoc id result))
    (cond
      (:orbit.ui.reflect/info metadata) (reflect new-state)
      (:orbit.ui/lazy metadata) (lazy new-state)
      (:orbit.shadow/error metadata) (shadow-errors new-state)
      (:tango/interactive metadata) (int/interactive new-state)
      (:tango/wrapped-error metadata) (wrapped-error new-state)
      (:tango/generic-stacktrace metadata) (generic-stacktrace new-state)
      (instance? serializer/RawData @result) (raw-data new-state)
      (serializer/tagged-literal-with-meta? @result) (tagged new-state)
      (map? @result) (as-map new-state)
      (vector? @result) (as-vector new-state)
      (set? @result) (as-set new-state)
      (coll? @result) (as-list new-state)
      :else (leaf new-state))))

(defn for-result [result editor-state]
  (r/with-let [local-state (r/atom {})
               patches (atom {})
               result (if (:error result)
                        (assoc result :result ^:tango/wrapped-error [(:error result)])
                        result)
               result-a (r/atom result)
               new-eql (helpers/prepare-new-eql editor-state)
               new-editor-state (atom (update @editor-state :editor/features assoc
                                             :eql new-eql
                                             :original-eql (-> @editor-state
                                                               :editor/features
                                                               :eql)))]
    ^{:patches patches}
    [as-html {:eval-result result-a
              :editor-state new-editor-state
              :patches patches
              :state local-state
              :root? true}]))
