(ns spock.swi
  (:require [instaparse.core :as insta]
            [spock.commons :as commons]
            [clojure.walk :as walk]
            [clojure.string :as str]))

(def parse-prolog
  (insta/parser
    "
<terms> = term (<',' ' '?> term)*
<term> = equality | atom | var | number | boolean | structure | list | string | weird-thing
equality = var <' = '>  term
list = <'[]'> | <'['> term (<',' ' '?> term)*  <']'>
weird-thing = #\"<.*?>\\(.*?\\)\"
structure = atom <'('> terms <')'>
string = <'\"'> (#\"[^\\\"]+\" | '\\\"')* <'\"'>
atom = atom-prefix letter* | \"'\" atom-sequence* \"'\"
<atom-sequence> = #\"(\\\\'|[^'])\"
var = var-prefix letter*
<atom-prefix> = #\"[a-z]\"
<letter> = #\"[\\d\\w_]\"
<var-prefix> = #\"[A-Z_]\"
number = #\"\\d+(\\.\\d+)?\"
boolean = 'true' | 'false'
"))

(defmulti parse-out first)
(defmethod parse-out :equality [[_ var thing]] [(parse-out var) (parse-out thing)])
(defmethod parse-out :weird-thing [[_ thing]] thing)
(defmethod parse-out :list [[_ & params]] (mapv parse-out params))
(defmethod parse-out :structure [[_ & structs]] (map parse-out structs))
(defmethod parse-out :string [[_ & str]] (str/join "" str))
(defmethod parse-out :atom [[_ & var]]
  (let [var-name (str/join "" var)]
    (-> var-name
        str/lower-case
        (str/replace-first #"^'" "")
        (str/replace-first #"'$" "")
        (str/replace #"_" "-")
        symbol)))

(defmethod parse-out :var [[_ & var]]
  (let [var-name (str/join "" var)]
    (-> var-name
        str/lower-case
        (str/replace #"_" "-")
        keyword)))

(defmethod parse-out :number [[_ num]] (js/parseInt num))
(defmethod parse-out :boolean [[_ b]] (= b "true"))

(def decoder (js/TextDecoder.))
(defn- normalize-stdout [out]
  (let [array (js/Uint8Array.from out)]
    (-> (.decode decoder array)
        (str/replace #"(\n|\s)+" " ")
        str/trim
        (str/replace #"\.$" ""))))

(def ^:private encoder (js/TextEncoder.))
(defn stdin-code [streams]
  (let [{:keys [pos code out next?]} @streams]
    (if (< pos (.-length code))
      (let [code (aget code pos)]
        (swap! streams update :pos inc)
        code)
      (when next?
        (swap! streams
               (fn [streams]
                 (-> streams
                     (assoc :code (. encoder encode "n")
                            :pos 0
                            :out #js []
                            :next? false)
                     (update :parsed conj (normalize-stdout (:out streams))))))
        (stdin-code streams)))))

(declare to-prolog)
(defn stdout-code [streams char]
  (let [out (:out @streams)
        as-str (js/String.fromCharCode char)]
    (.push out char)
    (swap! streams assoc :next? (and (not= as-str ".")
                                     (not= as-str "\n")))))

(defn prepare-run [streams ^js module]
  (.. module -FS (init (partial stdin-code streams)
                       (partial stdout-code streams))))


(defn- run-fn [{:keys [kind impl]} params]
  (case kind
    :normal (let [args (pop params)
                  res (apply impl params)]
              (conj args res))
    :unifier (apply impl params)))

(defn fake-dev [streams]
  #js {:open (constantly 1)
       :close (constantly nil)
       :read (fn [stream buffer offset length position]
               (if-let [str-val (-> @streams :to-deliver first)]
                 (let [val (.encode encoder str-val)
                       size (.-length val)]
                   (swap! streams update :to-deliver subvec 1)
                   (dotimes [n size]
                            (let [buffer-n (+ offset n)]
                              (aset buffer buffer-n (aget val n))))
                   size)
                 0))
       :write (fn [stream buffer offset length position]
                (let [res (.decode decoder (. buffer subarray offset, (+ offset length)))
                      [_ [fn-name & params]] (-> res
                                                 str/trimr
                                                 parse-prolog
                                                 first
                                                 parse-out)
                      params (vec params)
                      fun (-> @streams :functions (get fn-name))
                      function-result (try
                                        (-> (run-fn fun params)
                                            to-prolog
                                            (str "."))
                                        (catch :default e
                                          (prn :ERROR e)))]
                  (swap! streams update :to-deliver conj function-result)
                  length))})

(defn create-runtime!
  ([constructor] (create-runtime! constructor {}))
  ([constructor opts]
   (let [streams (atom {:pos 0
                        :code "A = 10."
                        :out #js []
                        :parsed []
                        :to-deliver []
                        :functions {}})]
     (. (constructor (-> {:arguments ["swipl" "-q" "-x" "/src/wasm-preload/boot.prc" "--nosignals"]}
                         (merge opts)
                         (assoc :preRun #js [(partial prepare-run streams)]
                                :printErr #(swap! streams update :error str % "\n"))
                         clj->js))
       (then (fn [^js r]
               (.. r -FS (mkdirTree "wasm-preload/library"))
               (.. r -FS (mkdirTree "/tmp"))
               (let [id (.. r -FS (makedev 88, 0))]
                 (.. r -FS (registerDevice id (fake-dev streams)))
                 (.. r -FS (mkdev "/dev/cljs-interop" id)))
               #js {:runtime r
                    :streams streams}))))))

(defn use-library [^js runtime lib-name contents]
  (.. runtime -runtime -FS (writeFile (str "wasm-preload/library/" lib-name ".pl")
                                      contents))
  (.. runtime -runtime -prolog (call_string (str "use_module(library(" lib-name "))"))))

(defn query! [^js swi code]
  (let [streams (.-streams swi)]
    (swap! streams assoc
           :to-deliver []
           :code (. encoder encode code)
           :pos 0
           :out #js []
           :parsed []
           :error nil)
    (.. swi -runtime -prolog (call_string "break"))
    (if-let [error (:error @streams)]
      (throw (ex-info "Query failed!" {:error error}))
      (->> (-> @streams :out normalize-stdout)
           (conj (:parsed @streams))
           (filter identity)))))

(defprotocol AsProlog (to-prolog [this]))

(defn- as-atom [unparsed]
  (str "'" (str/replace (str unparsed) #"\\" "\\\\") "'"))

(defn- as-struct [unparsed]
  (let [[head & tail] (commons/normalize-struct unparsed)]
    (str (as-atom head)
         "("
         (->> tail
              (map to-prolog)
              (str/join ", "))
         ")")))

(extend-protocol AsProlog
  string
  (to-prolog [this] (pr-str this))

  boolean
  (to-prolog [this] (pr-str this))

  number
  (to-prolog [this] (str this))

  Keyword
  (to-prolog [this] (-> this name (str/replace-first #"." (fn [s] (str/upper-case s)))))

  Symbol
  (to-prolog [this] (as-atom this))

  object
  (to-prolog [this]
    (cond
      (vector? this) (let [[bef aft] (split-with #(not= % '&) this)]
                       (str "["
                            (if (-> aft count (= 2))
                              (str (->> bef (map to-prolog) (str/join ", "))
                                   " | " (->> aft last to-prolog))
                              (->> this (map to-prolog) (str/join ", ")))
                            "]"))

      (let [mapped (map to-prolog this)])
      (str "["
           (->> this
                (map to-prolog)
                (str/join ", "))
           "]")
      (list? this) (as-struct this)
      :else (throw (ex-info "Unsupported Object" {:type (type this)})))))

(defrecord SWI [^js runtime]
  commons/SWIWrapper
  (call! [_ query] (.. runtime -runtime -prolog (call_string (to-prolog query)))))

(defn with-rules [runtime rules]
  (commons/with-rules (->SWI runtime) rules))

(defn assert-rules [runtime rules]
  (commons/assert-rules (->SWI runtime) rules))

(defn- get-val-for-key [key vars]
  (let [val (get vars key ::missing)]
    (cond
      (= val ::missing) [::missing key]
      (keyword? val) (get-val-for-key val vars)
      :else [val])))

(defn- unify-vars [vars]
  (if (some keyword? (vals vars))
    (->> vars
         (mapcat (fn [[k v]]
                   (let [[new-val possible-k] (get-val-for-key k vars)]
                     (if (= ::missing new-val)
                       [[k :_] [possible-k :_]]
                       [[k new-val]]))))
         (into {}))
    vars))

(defn- from-prolog [result-strings]
  (try
    (->> result-strings
         (map (fn [one-result]
                (try
                  (->> one-result
                       parse-prolog
                       (reduce #(let [parsed (parse-out %2)]
                                  (cond
                                    (vector? parsed) (let [[k v] parsed]
                                                       (if (and (keyword? k) (-> k name (str/starts-with? "-")))
                                                         %1
                                                         (assoc %1 k v)))
                                    parsed (conj %1 {})))
                               {})
                       unify-vars)
                  (catch :default e
                    (println "Error parsing:" e)
                    (println "Original:" result-strings)
                    :invalid-output))))
         (filter identity))
    (catch :default e
      (println "Error parsing:" e)
      :invalid-output)))

(defn- rewrite-cljs-funs [functions query]
  (walk/postwalk (fn [form]
                   (if (and (list? form) (-> form first functions))
                     (let [w-stream (keyword (str (gensym "_w")))
                           r-stream (keyword (str (gensym "_r")))
                           args-as-is (-> form rest vec)]
                       `(~'and
                          (~'open "/dev/cljs-interop" ~'write ~w-stream)
                          (~'write ~w-stream ~(to-prolog (list 'run-cljs form)))
                          (~'write ~w-stream ", ")
                          (~'writeln ~w-stream ~args-as-is)
                          (~'close ~w-stream)
                          (~'open "/dev/cljs-interop" ~'read ~r-stream)
                          (~'read ~r-stream ~args-as-is)
                          (~'close ~r-stream)))
                     form))
                 query))

(defn solve [{:keys [runtime] :as opts} query]
  (let [rewroten-query (rewrite-cljs-funs (-> runtime .-streams deref :functions) query)
        [query from-prolog] (commons/prepare-solve from-prolog opts rewroten-query)
        as-str (-> query to-prolog (str "."))]
    (->> as-str
         (query! runtime)
         from-prolog)))

(defn bind-function! [runtime fn-name fn-implementation]
  (swap! (.-streams runtime) assoc-in [:functions (-> fn-name name symbol)]
         {:kind :normal
          :impl fn-implementation})
  nil)

(defn bind-unifier! [runtime fn-name fn-implementation]
  (swap! (.-streams runtime) assoc-in [:functions (-> fn-name name symbol)]
         {:kind :unifier
          :impl fn-implementation})
  nil)
