(ns orbit.serializer
  (:require [clojure.edn :as edn]))

;; Not using protocols because this will be used in the REPL, so it needs to be easily injectable
(defn to-ebn [object#]
  (let [to-map# (fn [object#]
                  (let [vals# (for [kv# object#
                                    obj# kv#]
                                (to-ebn obj#))]
                    (str "a" (apply str vals#) "e")))

        to-raw# (fn [object#] (let [as-str# (pr-str object#)
                                    meta# (comment "RawMeta")]
                                (str (when meta# (str "M" (to-ebn meta#)))
                                     "R" (count as-str#) "e" as-str#)))
                                  ; (str "M" (to-ebn meta#)))
                                  ;
                                  ;  (to-ebn-no-meta# object#))
                                  ; (to-ebn))
                                ; #?@(:clj [(comment "Clojure")
                                ;           (str "R" (count as-str#) "e" as-str#)]
                                ;     :default [(str "R" (count as-str#) "e" as-str#)])))
        to-ebn-no-meta# (fn [object#]
                          (cond
                            (nil? object#) "n"
                            (= object# true) "u"
                            (= object# false) "f"
                            (= ##Inf object#) "i"
                            (= ##-Inf object#) "I"
                            #?@(:clj [(ratio? object#) (to-raw# object#)])
                            #?@(:bb []
                                :nbb []
                                :clj [(tagged-literal? object#) (str "#"
                                                                     (count (str (.tag object#)))
                                                                     "e"
                                                                     (str (.tag object#))
                                                                     (to-ebn (.form object#)))]
                                :cljs [(tagged-literal? object#) (str "#"
                                                                      (count (str (.-tag object#)))
                                                                      "e"
                                                                      (str (.-tag object#))
                                                                      (to-ebn (.-form object#)))])
                            (->> object# type pr-str (re-find (re-pattern "(?i)big.*(dec|int)")))
                            (to-raw# object#)

                            (= "##NaN" (pr-str object#)) "N"
                            #?@(:cljs [(instance? js/Error object#)
                                       (let [as-str# (pr-str {:cause (.-message object#)
                                                              :original-object object#
                                                              :trace (.-stack object#)})]
                                         (str "#5eerrorR" (count as-str#) "e" as-str#))])
                            (number? object#) (str "m" object# "m")
                            (keyword? object#) (str "k" (-> object# str count dec) "e" (-> object# str (subs 1)))
                            (string? object#) (str "s" (count object#) "e" object#)
                            (symbol? object#) (str "y" (-> object# str count) "e" object#)
                            (re-find (re-pattern "(?i)regex") (pr-str (type object#))) (let [s# (str object#)
                                                                                             #?@(:cljs [s# (subs s# 1 (-> s# count (- 1)))])]
                                                                                         (str "r" (count s#) "e" s#))
                            (vector? object#) (str "v" (apply str (map to-ebn object#)) "e")
                            (set? object#) (str "t" (apply str (map to-ebn object#)) "e")

                            (->> object# pr-str (re-find (re-pattern "^#[^:]+[ \\{\\(\\[]")))
                            (let [as-str# (pr-str object#)
                                  regex# (re-pattern "(?s)^#(.+?)([ \\{\\(\\[].*)")
                                  [_# tag# val#] (re-find regex# as-str#)]
                              (str "#" (count tag#) "e" tag#
                                   (cond
                                     (map? object#) (to-map# object#)
                                     (clojure.string/starts-with? val# " ") (-> val# (subs 1) symbol to-raw#)
                                     :else (to-raw# (symbol val#)))))

                            (map? object#) (to-map# object#)
                            (coll? object#) (str "l" (apply str (map to-ebn object#)) "e")
                            :else (to-raw# object#)))
        to-meta# (fn [meta# object#]
                   (str "M"
                        (to-ebn meta#)
                        (to-ebn-no-meta# object#)))
        meta# (meta object#)]
    (cond
      meta# (to-meta# meta# object#)
      :else (to-ebn-no-meta# object#))))

(declare deserialize)
(defn- number-deserializer [string sofar]
  (let [[full-match number terminator] (re-find #"(?i)^([\d\.\+\-e]*)(m)" string)]
    (if terminator
      (let [final (str sofar number)]
        {:result (if (re-find #"\." final)
                   #?(:clj (Double/parseDouble final) :cljs (js/parseFloat final))
                   #?(:clj (Long/parseLong final) :cljs (js/parseInt final)))
         :rest (subs string (count full-match))})
      #(number-deserializer % (str sofar string)))))

(defn- coll-deserializer [string previous-result sofar final-fn]
  (let [result-from-deserialize (delay (previous-result string))]
    (cond
      (-> string first (= \e))
      {:result (final-fn (persistent! sofar))
       :rest (subs string 1)}

      (fn? @result-from-deserialize)
      (fn inner-col-deserializer [%]
        (coll-deserializer %
                           @result-from-deserialize
                           sofar
                           final-fn))

      :final-parser-result
      (recur (:rest @result-from-deserialize)
        deserialize
        (conj! sofar (:result @result-from-deserialize))
        final-fn))))

(defn- contents-counted-obj-deserializer [final-fn string size sofar]
  (let [new-so-far (str sofar string)]
    (if (-> new-so-far count (>= size))
      {:result (final-fn (subs new-so-far 0 size))
       :rest (subs new-so-far size)}
      #(contents-counted-obj-deserializer final-fn % size new-so-far))))

(defn- size-counted-obj-deserializer [final-fn string size-sofar]
  (let [[full got control] (re-find #"^(\d*)(e)" string)]
    (if control
      (contents-counted-obj-deserializer final-fn (subs string (count full))
                                    #?(:clj (Long/parseLong (str size-sofar got))
                                       :cljs (js/parseInt (str size-sofar got)))
                                    "")
      #(size-counted-obj-deserializer final-fn % (str size-sofar string)))))

(defn- compose-obj-deserializer [tag res final-fn]
  (if (fn? res)
    #(compose-obj-deserializer tag (res %) final-fn)
    (update res :result #(final-fn tag %))))

(defn- compose-deserializer [res final-fn]
  (if (fn? res)
    #(compose-deserializer (res %) final-fn)
    (compose-obj-deserializer (:result res)
                              (deserialize (:rest res))
                              final-fn)))

(defrecord RawData [data])
(deftype TaggedLiteralWithMeta [tag form meta]
  #?@(:clj [clojure.lang.IObj
            (withMeta [_ new-meta] (TaggedLiteralWithMeta. tag form new-meta))
            (meta [_] meta)
            (toString [this] (pr-str this))]
      :cljs [IWithMeta
             (-with-meta [_ new-meta] (TaggedLiteralWithMeta. tag form new-meta))
             IMeta
             (-meta [_] meta)
             IPrintWithWriter
             (-pr-writer [_ writer opts]
               (-write writer (str "#" tag " " (pr-str form))))
             Object
             (toString [_]
               (str "#" tag " " (pr-str form)))]))

(defn tagged-literal-with-meta? [obj]
  (or (instance? TaggedLiteralWithMeta obj)
      (tagged-literal? obj)))

(defn tagged-literal-with-meta [tag form]
  (TaggedLiteralWithMeta. tag form nil))

#?(:clj
   (defmethod print-method TaggedLiteralWithMeta [x writer]
     (.write writer (str "#" (.-tag x) " "))
     (print-method (.-form x) writer)))

(defn- from-raw [some-string-data]
  (try
    #?(:clj
       (edn/read-string {:default tagged-literal-with-meta} some-string-data)
       :cljs
       (cond
         (-> some-string-data last #{"M" "N"}) (->RawData some-string-data)
         (re-matches #"\\." some-string-data) (->RawData some-string-data)
         :else (edn/read-string {:default tagged-literal-with-meta} some-string-data)))
    (catch #?(:clj Throwable :cljs :default) _
      (->RawData some-string-data))))

(defn deserialize [string]
  (let [rest (some-> string not-empty (subs 1))]
    (case (-> string first str)
      "n" {:result nil :rest rest}
      "u" {:result true :rest rest}
      "f" {:result false :rest rest}
      "i" {:result ##Inf :rest rest}
      "I" {:result ##-Inf :rest rest}
      "N" {:result ##NaN :rest rest}
      "m" (number-deserializer rest "")
      "v" (coll-deserializer rest deserialize (transient []) identity)
      "l" (coll-deserializer rest deserialize (transient []) #(apply list %))
      "t" (coll-deserializer rest deserialize (transient []) set)
      "a" (coll-deserializer rest deserialize (transient []) #(apply array-map %))
      "s" (size-counted-obj-deserializer str rest "")
      "k" (size-counted-obj-deserializer keyword rest "")
      "r" (size-counted-obj-deserializer re-pattern rest "")
      "y" (size-counted-obj-deserializer symbol rest "")
      "R" (size-counted-obj-deserializer from-raw rest "")
      "#" (compose-deserializer (size-counted-obj-deserializer symbol rest "")
                                tagged-literal-with-meta)
      "M" (compose-deserializer (deserialize rest)
                                #(with-meta %2 %1))
      "" deserialize)))
