(ns orbit.serializer
  (:require [clojure.edn :as edn]))

;; Not using protocols because this will be used in the REPL, so it needs to be easily injectable
(defn to-ebn [object#]
  (let [to-map# (fn [object#]
                  (let [vals# (for [kv# object#
                                    obj# kv#]
                                (to-ebn obj#))]
                    (str "a" (apply str vals#) "e")))
        to-raw# (fn [object#] (let [as-str# (pr-str object#)]
                                (str "R" (count as-str#) "e" as-str#)))
        to-ebn-no-meta# (fn [object#]
                          (cond
                            (nil? object#) "n"
                            (= object# true) "u"
                            (= object# false) "f"
                            (= ##Inf object#) "i"
                            (= ##-Inf object#) "I"
                            (= "##NaN" (pr-str object#)) "N"
                            #?@(:cljs [(instance? js/Error object#)
                                       (let [as-str# (pr-str {:cause object#
                                                              :stacktrace (.-stack object#)})]
                                         (str "R" (count as-str#) "e" as-str#))])
                            (number? object#) (str "m" object# "m")
                            (keyword? object#) (str "k" (-> object# str count dec) "e" (-> object# str (subs 1)))
                            (string? object#) (str "s" (count object#) "e" object#)
                            (symbol? object#) (str "y" (-> object# str count) "e" object#)
                            (re-find #"(?i)regex" (pr-str (type object#))) (let [s# (str object#)
                                                                                 s# (subs s# 1 (-> s# count (- 1)))]
                                                                             (str "r" (count s#) "e" s#))
                            (vector? object#) (str "v" (apply str (map to-ebn object#)) "e")
                            (set? object#) (str "t" (apply str (map to-ebn object#)) "e")

                            (->> object# pr-str (re-find #"^#.+[ \{\(\[]"))
                            (let [as-str# (pr-str object#)
                                  regex# #"(?s)^#(.+?)([ \{\(\[].*)"
                                  [_# tag# val#] (re-find regex# as-str#)]
                              (str "#" (count tag#) "e" tag#
                                   (cond
                                     (map? object#) (to-map# object#)
                                     (clojure.string/starts-with? val# " ") (-> val# (subs 1) symbol to-raw#)
                                     :else (to-raw# (symbol val#)))))

                            (map? object#) (to-map# object#)
                            (coll? object#) (str "l" (apply str (map to-ebn object#)) "e")
                            :else (to-raw# object#)))
        to-meta# (fn [meta# object#]
                   (str "M"
                        (to-ebn meta#)
                        (to-ebn-no-meta# object#)))
        meta# (meta object#)]
    (cond
      meta# (to-meta# meta# object#)
      :else (to-ebn-no-meta# object#))))

(declare deserialize)
(defn- number-deserializer [string sofar]
  (let [[full-match number terminator] (re-find #"^([\d\.\+\-e]*)(m)" string)]
    (if terminator
      (let [final (str sofar number)]
        {:result (if (re-find #"\." final)
                   (js/parseFloat final)
                   (js/parseInt final))
         :rest (subs string (count full-match))})
      #(number-deserializer % (str sofar string)))))

(defn- coll-deserializer [string previous-result sofar final-fn]
  (let [result-from-deserialize (delay (previous-result string))]
    (cond
      (-> string first (= "e"))
      {:result (final-fn (persistent! sofar))
       :rest (subs string 1)}

      (fn? @result-from-deserialize)
      #(coll-deserializer %
                          @result-from-deserialize
                          sofar
                          final-fn)

      :final-parser-result
      (recur (:rest @result-from-deserialize)
        deserialize
        (conj! sofar (:result @result-from-deserialize))
        final-fn))))

(defn- contents-counted-obj-deserializer [final-fn string size sofar]
  (let [new-so-far (str sofar string)]
    (if (-> new-so-far count (>= size))
      {:result (final-fn (subs new-so-far 0 size))
       :rest (subs new-so-far size)}
      #(contents-counted-obj-deserializer final-fn % size new-so-far))))

(defn- size-counted-obj-deserializer [final-fn string size-sofar]
  (let [[full got control] (re-find #"^(\d*)(e)" string)]
    (if control
      (contents-counted-obj-deserializer final-fn (subs string (count full))
                                    (js/parseInt (str size-sofar got))
                                    "")
      #(size-counted-obj-deserializer final-fn % (str size-sofar string)))))

(defn- compose-obj-deserializer [tag res final-fn]
  (if (fn? res)
    #(compose-obj-deserializer tag (res %) final-fn)
    (update res :result #(final-fn tag %))))

(defn- compose-deserializer [res final-fn]
  (if (fn? res)
    #(compose-deserializer (res %) final-fn)
    (compose-obj-deserializer (:result res)
                              (deserialize (:rest res))
                              final-fn)))

(defrecord RawData [data])
(defrecord TaggedLiteral2 [tag form])

(defn- from-raw [some-string-data]
  (try
    (edn/read-string {:default tagged-literal} some-string-data)
    (catch :default _
      (->RawData some-string-data))))

(defn deserialize [string]
  (let [rest (subs string 1)]
    (case (first string)
      "n" {:result nil :rest rest}
      "u" {:result true :rest rest}
      "f" {:result false :rest rest}
      "i" {:result ##Inf :rest rest}
      "I" {:result ##-Inf :rest rest}
      "N" {:result ##NaN :rest rest}
      "m" (number-deserializer rest "")
      "v" (coll-deserializer rest deserialize (transient []) identity)
      "l" (coll-deserializer rest deserialize (transient []) #(apply list %))
      "t" (coll-deserializer rest deserialize (transient []) set)
      "a" (coll-deserializer rest deserialize (transient []) #(apply hash-map %))
      "s" (size-counted-obj-deserializer str rest "")
      "k" (size-counted-obj-deserializer keyword rest "")
      "r" (size-counted-obj-deserializer re-pattern rest "")
      "y" (size-counted-obj-deserializer symbol rest "")
      "R" (size-counted-obj-deserializer from-raw rest "")
      "#" (compose-deserializer (size-counted-obj-deserializer symbol rest "")
                                tagged-literal)
      "M" (compose-deserializer (deserialize rest)
                                #(with-meta %2 %1))
      nil deserialize)))

(extend-type TaggedLiteral
  IMeta
  (-meta [this] (aget this "--meta"))

  IWithMeta
  (-with-meta [this meta]
    (doto (tagged-literal (.-tag this) (.-form this))
          (aset "--meta" meta))))
