(ns duck-repled.repl-resolvers
  (:require [clojure.string :as str]
            [duck-repled.connect :as connect]
            [com.wsscode.pathom3.connect.operation :as pco]
            [orbit.evaluation :as eval]
            [duck-repled.template :refer [template]]
            [duck-repled.editor-helpers :as helpers]
            [clojure.walk :as walk]
            [promesa.core :as p]))

#_
(connect/defresolver get-right-repl [{:repl/keys [kind evaluators]}]
  {::pco/output [:repl/evaluator :repl/clj]}

  (prn :KIND kind)
  (let [{:keys [clj cljs]} evaluators]
    (cond
      (not= kind :cljs) {:repl/evaluator clj :repl/clj clj}
      (nil? clj) {:repl/clj cljs :repl/evaluator cljs}
      :else {:repl/clj clj :repl/evaluator cljs})))

(connect/defresolver repl-eval [env inputs]
  {::pco/input [:repl/evaluator :text/contents
                (pco/? :repl/aux?) (pco/? :repl/kind)
                (pco/? :editor/filename) (pco/? :repl/namespace) (pco/? :text/range)]
   ::pco/output [:repl/result :repl/error]}

  (p/let [{:repl/keys [evaluator namespace aux? kind]
           :text/keys [contents range]
           :editor/keys [filename]}
          inputs
          params (pco/params env)
          option-kind (when kind (keyword (name kind) (if aux? "aux" "eval")))
          opts (cond-> (dissoc params :repl/template)
                       option-kind (assoc-in [:options :kind] option-kind)
                       namespace (assoc :namespace namespace)
                       filename (assoc :file filename)
                       range (-> (update :row #(or % (-> range first first)))
                                 (update :col #(or % (-> range first second)))))
          code (if-let [t (:repl/template params)]
                 (template t {:repl/code (symbol contents)})
                 contents)
          result (eval/evaluate evaluator code opts)]
    (if (:error result)
      {:repl/error result}
      {:repl/result result})))

(defn- extract-right-var [current-var contents]
  (let [contents (or (:text/contents current-var) contents)
        [_ var] (helpers/current-var (str contents) [0 0])]
    (when (and var (= var contents))
      contents)))

(connect/defresolver fqn-var
  [{:keys [repl/namespace text/current-var text/contents repl/evaluator
           repl/kind repl/aux?]}]
  {::pco/input [:repl/namespace :repl/evaluator
                (pco/? :repl/kind) (pco/? :repl/aux?)
                (pco/? :text/current-var) (pco/? :text/contents)]
   ::pco/output [:var/fqn]}

  (when-let [contents (extract-right-var current-var contents)]
    (p/let [option-kind (when kind (keyword (name kind) (if aux? "aux" "eval")))
            {:keys [result]} (eval/evaluate evaluator
                                            (str "`" contents)
                                            {:namespace namespace
                                             :options {:kind option-kind}})]
      {:var/fqn result})))

(connect/defresolver meta-for-var
  [{:keys [repl/namespace repl/evaluator repl/kind
           text/current-var text/contents]}]
  {::pco/input [:repl/namespace :repl/evaluator
                (pco/? :text/current-var) (pco/? :text/contents)]
   ::pco/output [:var/meta]
   ::pco/priority 1}

  (when-let [contents (extract-right-var current-var contents)]
    (p/let [option-kind (keyword (name kind) "aux")
            code (template `(meta ::current-var)
                            {::current-var (->> contents
                                                (str "#'")
                                                symbol)})
            {:keys [result error]} (eval/evaluate evaluator
                                                  code
                                                  {:namespace namespace
                                                   :options {:kind option-kind}})]
      (cond
        result {:var/meta result}
        (= :cljs kind) (p/then (eval/evaluate evaluator
                                              code
                                              {:namespace namespace
                                               :options {:kind :clj/aux}})
                               (fn [r] {:var/meta (:result r)}))))))

; TODO: Somehow, test this
(pco/defresolver spec-for-var [{:keys [var/fqn repl/evaluator]}]
  {::pco/output [:var/spec]}

  (p/let [res (eval/evaluate evaluator "(require 'clojure.spec.alpha)" {})]
    (when-not (:error res)
      (p/let [{:keys [result]}
              (eval/evaluate
               evaluator
               (template `(let [s# (clojure.spec.alpha/get-spec ' ::fqn)
                                fun# #(some->> (% s) clojure.spec.alpha/describe)]
                            (when s#
                              (->> [:args :ret :fn]
                                   (map (juxt identity fun#))
                                   (filter second)
                                   (into {}))))
                         {::fqn fqn})
               {})]
        (when result {:var/spec result})))))

(pco/defresolver doc-for-var [{:var/keys [fqn meta spec]}]
  {::pco/input [:var/fqn :var/meta (pco/? :var/spec)]
   ::pco/output [:var/doc]}

  {:var/doc
   (str "-------------------------\n"
        fqn "\n"
        (:arglists meta) "\n  "
        (:doc meta)
        (when (map? spec)
          (cond-> "\nSpec\n"
                  (:args spec) (str "  args: " (pr-str (:args spec)) "\n")
                  (:ret spec) (str "  ret: " (pr-str (:ret spec)) "\n")
                  (:fn spec) (str "  fn: " (pr-str (:fn spec))))))})

(def resolvers [repl-eval fqn-var
                meta-for-var spec-for-var doc-for-var])
