(ns duck-repled.autocomplete-resolvers
  (:require [duck-repled.connect :as connect]
            [com.wsscode.pathom3.connect.operation :as pco]
            [clojure.string :as str]
            [duck-repled.template :as t]
            [promesa.core :as p]
            [duck-repled.repl-protocol :as repl]))

(def ^:private valid-prefix #"/?([a-zA-Z0-9\-.$!?\/><*=\?_:]+)")
(def ^:private re-char-escapes
  (->> "\\.*+|?()[]{}$^"
       set
       (map (juxt identity #(str "\\" %)))
       (into {})))
(defn- re-escape [prefix] (str/escape (str prefix) re-char-escapes))

(defn- make-prefix-re [contents]
  (->> contents (re-seq valid-prefix) last last re-escape (str "^") re-pattern))

(connect/defresolver clojure-complete [{:keys [text/contents repl/evaluator
                                               repl/kind repl/namespace]}]
  {::pco/output [:completions/var]}
  (when-not (= kind :cljs)
    (p/let [prefix (make-prefix-re contents)
            cmd `(let [collect# #(map (comp str first) (%1 %2))
                       refers# (collect# clojure.core/ns-map *ns*)
                       from-ns# (->> (clojure.core/ns-aliases *ns*)
                                     (mapcat (fn [[k# v#]]
                                               (map #(str k# "/" %)
                                                    (collect# ns-publics v#)))))]
                   (->> refers#
                        (concat from-ns#)
                        (filter #(re-find :prefix-regexp %))))
            res (repl/eval evaluator
                           (t/template cmd {:prefix-regexp prefix})
                           {:namespace namespace})]
      {:completions/var (mapv (fn [res]
                                {:text/contents res
                                 :completion/type :function})
                              (:result res))})))

(connect/defresolver clojure-keyword [{:keys [text/contents repl/evaluator repl/kind]
                                       ns :repl/namespace}]
  {::pco/output [:completions/keyword]}
  (when-not (= kind :cljs)
    (p/let [prefix (make-prefix-re contents)
            cmd `(let [^java.lang.reflect.Field field# (.getDeclaredField clojure.lang.Keyword "table")]
                   (.setAccessible field# true)
                   (->> (.get field# nil)
                        (map #(.getKey %))))
            {:keys [result]} (repl/eval evaluator (t/template cmd {}))
            imports (when (str/starts-with? contents "::")
                      (repl/eval evaluator (t/template `(map #(mapv str %) (clojure.core/ns-aliases *ns*))
                                                       {})
                                 {:namespace ns}))
            all-nses (cond-> result
                             imports (concat
                                      (for [[alias nss] (:result imports)
                                            one-ns result
                                            :when (symbol? one-ns)
                                            :let [row-ns (namespace one-ns)]
                                            :when (= row-ns nss)]
                                        (str ":" alias "/" (name one-ns)))))]
      {:completions/keyword
       (->> all-nses
            (map #(str ":" %))
            (filter #(re-find prefix %))
            (mapv (fn [elem] {:text/contents elem :completion/type :keyword})))})))

(def ^:private get-completions-tpl
  `(let [curr-ns# (-> ::env
                      :cljs.analyzer/namespaces
                      (get '::namespace))
         filter-map# (fn [e#] (->> e#
                                   (filter #(->> % first str (re-find ::prefix)))
                                   (map (fn [[k# v#]]
                                          {:completion/type (cond
                                                              (:macro v#) :macro
                                                              (:fn-var v#) :function
                                                              :else :var)
                                           :text/contents (str k#)}))))]
     (concat (-> curr-ns# :defs filter-map#)
             (-> curr-ns# :macros filter-map#)
             (->> curr-ns# :requires
                  (filter #(->> % first str (re-find ::prefix)))
                  (map (fn [[k#]] {:text/contents (str k#)
                                   :completion/type :namespace})))
             (->> curr-ns# :uses
                  (filter #(->> % first str (re-find ::prefix)))
                  (map (fn [[k#]] {:text/contents (str k#)
                                   :completion/type :other})))
             (->> curr-ns# :use-macros
                  (filter #(->> % first str (re-find ::prefix)))
                  (map (fn [[k#]] {:text/contents (str k#)
                                   :completion/type :macro}))))))

(defn- ns-alias->fqn [repl cljs-env ns-name ns-alias]
  (p/then (repl/eval repl (t/template `(-> ::env
                                           :cljs.analyzer/namespaces
                                           (get '::namespace)
                                           :requires
                                           (get '::namespace-part))
                                      {::env (symbol cljs-env)
                                       ::namespace ns-name
                                       ::namespace-part (symbol ns-alias)}))
          :result))

(defn- get-ns-completions! [clj contents cljs-env ns-name]
  (p/let [[ns-part var-part] (str/split contents #"/")
          ; res (repl/eval clj (t/template `(-> ::env
          ;                                     :cljs.analyzer/namespaces
          ;                                     (get '::namespace)
          ;                                     :requires
          ;                                     (get '::namespace-part))
          ;                                {::env (symbol cljs-env)
          ;                                 ::namespace ns-name
          ;                                 ::namespace-part (symbol ns-part)}))
          fqn-ns (ns-alias->fqn clj cljs-env ns-name ns-part)
          comp (repl/eval clj (t/template get-completions-tpl
                                          {::env (symbol cljs-env)
                                           ::namespace fqn-ns
                                           ::prefix (make-prefix-re var-part)}))]
    (->> comp
         :result
         (map (fn [e] (update e :text/contents #(str ns-part "/" %))))
         (hash-map :result))))

(connect/defresolver cljs-complete [{:keys [text/contents repl/clj
                                            repl/kind repl/cljs-env]
                                     ns-name :repl/namespace}]
  {::pco/output [:completions/var]}
  (when (= kind :cljs)
    (p/let [prefix (make-prefix-re contents)
            cmd (t/template get-completions-tpl
                            {::env (symbol cljs-env)
                             ::namespace ns-name
                             ::prefix prefix})
            is-ns-prefix? (re-find #"/" contents)
            additional (if is-ns-prefix?
                         (get-ns-completions! clj contents cljs-env ns-name)
                         (repl/eval clj (t/template get-completions-tpl
                                                    {::env (symbol cljs-env)
                                                     ::namespace 'cljs.core
                                                     ::prefix prefix})))
            {:keys [result]} (repl/eval clj cmd)]
      {:completions/var (-> result
                            (concat (:result additional))
                            vec)})))

(defn- norm-prefix-regexp [repl contents curr-ns get-ns!]
  (if (re-find #"^::" contents)
    (if (re-find #"/" contents)
      (p/let [[ns-name var-name] (str/split contents #"/")
              fqn-ns (get-ns! (subs ns-name 2))]
        (if fqn-ns
          [(make-prefix-re (str ":" fqn-ns "/" var-name)) #(str ns-name "/" (name %))]
          [#"::::" identity]))
      [(make-prefix-re (str ":" curr-ns "/" (subs contents 2))) #(str "::" (name %))])
    [(make-prefix-re contents) str]))

(connect/defresolver cljs-kw-complete [{:keys [text/contents repl/clj
                                               repl/kind repl/cljs-env]
                                        ns-name :repl/namespace}]
  {::pco/output [:completions/keyword]}
  (when (= kind :cljs)
    (p/let [tpl `(->> ::cljs-env
                      :cljs.analyzer/constant-table
                      (map first)
                      (filter #(->> % str (re-find ::norm-prefix))))
            [prefix-re norm-fn] (norm-prefix-regexp clj
                                                    contents
                                                    ns-name
                                                    #(ns-alias->fqn clj cljs-env ns-name %))
            matches (repl/eval clj (t/template tpl {::cljs-env (symbol cljs-env)
                                                    ::norm-prefix prefix-re}))]
      (->> matches
           :result
           (mapv (fn [kw]
                   {:text/contents (norm-fn kw)
                    :completion/type :keyword}))
           (hash-map :completions/keyword)))))

(def resolvers [clojure-complete clojure-keyword
                cljs-complete cljs-kw-complete])
