(ns frame.core
  (:require [hx.react :as hx]
            [hx.hooks :as hooks]))

(defn- <-emitter []
  (let [id->handler (hooks/<-ref {})
        id->subscribers (hooks/<-ref {})]
    {::id->handler id->handler
     ::id->subscribers id->subscribers

     ::emit (fn emit [[id & params :as event]]
              (prn 'emitting)
              (let [handler (@id->handler id)
                    subscribers (@id->subscribers id)]
                (prn id)
                (doseq [subscriber subscribers]
                  (subscriber handler
                              event))))

     ::add-handler (fn add-handler [id handler]
                     (swap! id->handler assoc id handler))

     ::add-subscriber (fn add-subscriber [id sub]
                        (if (contains? id->subscribers id)
                          (swap! id->subscribers
                                 update id conj sub)
                          (swap! id->subscribers
                                 assoc id #{sub})))

     ::clear-id (fn clear-id [id]
                  (swap! id->handler dissoc id)
                  (swap! id->subscribers dissoc id))

     ::clear-subscriber (fn clear-subscriber [id subscriber]
                          (swap! id->subscribers
                                 update id disj subscriber))}))

(def frame-context (hx/create-context))

(hx/defnc Provider [{:keys [initial-db children]}]
  (let [app-db (hooks/<-ref initial-db)
        sub-emitter (<-emitter)
        event-id->handler (hooks/<-ref {})
        frame {::sub-add (::add-subscriber sub-emitter)
               ::sub-reg (::add-handler sub-emitter)
               ::sub-clear (::clear-subscriber sub-emitter)
               ::sub-handlers (::id->handler sub-emitter)
               ::sub-emit (::emit sub-emitter)
               ;; events
               ::event-handlers event-id->handler
               ::event-add (fn event-add [id h]
                             (swap! event-id->handler assoc id h))
               ::event-clear (fn event-clear [id]
                               (swap! event-id->handler dissoc id))
               ;; misc
               ::app-db app-db}]
    [:provider {:context frame-context
                :value frame}
     children]))

(defn <-reg-sub [id h]
  (let [frame (hooks/<-context frame-context)
        {:keys [::sub-reg]} frame]
    ;; should check if already exists
    (sub-reg id h)))

(defn <-sub [id]
  (let [frame (hooks/<-context frame-context)
        {:keys [::app-db ::sub-add ::sub-clear
                ::sub-handlers ::event-handlers]} frame
        sub-handler (@sub-handlers id)]
    (if (nil? sub-handler)
      (println (str "No subscription " id " found."))
      ;; memoize computation of initial state
      (let [[v u] (hooks/<-state (hooks/<-memo #(sub-handler @app-db)
                                               [app-db sub-handler]))]
        ;; this might happen too late w/ <-effect
        (hooks/<-effect
         (fn sub-add-effect []
           (let [sub (fn subscriber [sub-handler [ev-id & params :as event]]
                       (u (fn updater [_]
                            (let [ev-handler (@event-handlers ev-id)
                                  app-db' (ev-handler @app-db event)]
                              (prn app-db')
                              (sub-handler app-db)))))]
             (sub-add id sub)
             #(sub-clear sub))
           []))
        v))))

(defn <-reg-event [id h]
  (let [frame (hooks/<-context frame-context)
        {:keys [::event-add]} frame]
    ;; should check if already exists
    (event-add id h)))

(defn <-dispatcher []
  (let [frame (hooks/<-context frame-context)]
    (::sub-emit frame)))

(defn <-frame []
  {:<-sub <-sub
   :<-reg-sub <-reg-sub
   :<-reg-event <-reg-event
   :dispatch (<-dispatcher)})

(comment
  (contains? {:foo "bar"} :foo)
  (conj #{:foo} :bar))
