(ns puddle.core
  "Puddle is a simple object pool library for Clojure."
  (:require [clojure.core.async :as async]))

(set! *warn-on-reflection* true)

;; Helpers

(defn- throw-str [^String s]
  (throw (Error. s)))

(defn- assoc-not-existing
  "assoc k to v in the map m, but throw an Error with the provided message if m already contains k."
  [m k v ^String message]
  (when (contains? m k) (throw-str message))
  (assoc m k v))

(defn- dissoc-existing
  "dissoc k in the map m, but throw an Error with the provided message if m does not contain k."
  [m k ^String message]
  (when-not (contains? m k) (throw-str message))
  (dissoc m k))

(defmacro safe-thread
  "Only for internal use"
  [& body]
  `(async/thread
     (try ~@body
          (catch Throwable t#
            (println "puddle: unhandled exception in background thread!"
            (.printStackTrace t#)
            (System/exit 1))))))

;; Types

; Not for direct use.
(defrecord PooledObject
  [obj ; may be nil; underlying user objects are lazily instantiated
   ^long last-updated ; only meaningful if obj is not nil
   ])

; Not for direct use; use the exported functions of this library to create and use Pools.
(defrecord Pool
  [new-fn
   destroy-fn
   validate-fn
   max-age-ms

   allocated-ch ; buffered chan of previously created objects
   fresh-ch     ; buffered chan of unallocated objects
   borrowed     ; atom, map of obj -> PooledObject wrapping it, just the outstanding borrowed objects
   ])

;; Internals

(defn- expired? [^Pool pool ^PooledObject pooled-obj]
  ; only used from the evictor, which is only enabled if a max-age-ms is provided.
  (> (- (System/currentTimeMillis) (.last-updated pooled-obj))
      (.max-age-ms pool)))

(defn- destroy-object [^Pool pool ^PooledObject pooled-obj]
  ((.destroy-fn pool) (.obj pooled-obj))
  (let [pooled-obj (assoc pooled-obj
                          :last-updated 0
                          :obj nil)]
    (async/>!! (.fresh-ch pool) pooled-obj)))

(defn- run-evictor
  [^Pool pool delay-ms]
  (while true
    (loop []
      (let [^PooledObject pooled-obj (async/<!! (.allocated-ch pool))]
        (if-not (expired? pool pooled-obj) ; stop looping when we find the first non-expired pooled-obj
          (async/>!! (.allocated-ch pool) pooled-obj)
          (do
            (let [validate-fn (.validate-fn pool)]
              (if (and validate-fn (validate-fn (.obj pooled-obj)))
                (async/>!! (.allocated-ch pool) pooled-obj)
                (destroy-object pool pooled-obj))
              (recur))))))
    (Thread/sleep delay-ms)))

;; Public API

(defn new-pool
  "Creates and returns a new pool with the given factory function and size. When borrowing from the pool,
  previously allocated and returned objects will be given back, or if none are available, new objects will be
  created using new-fn, until the pool is at its maximum size (pool-size). It is an error if new-fn returns
  nil.

  The options map allows for a few more parameters to be set:
  - :destroy-fn
  - :max-age-ms
  - :validate-fn

  Objects only expire if :max-age-ms is given. On borrow!, objects that were last created/returned more than
  :max-age-ms ago are expired, and considered for destruction. If :validate-fn is not given, then expired
  objects are always destroyed; otherwise they are tested using the provided function and only destroyed if it
  returns false. If :destroy-fn is provided it is called with objects when they are to be destroyed."
  [new-fn pool-size opts]
  (let [{:keys [destroy-fn max-age-ms validate-fn]
         :or {destroy-fn (constantly nil) ; don't do anything
              }} opts]
    (when (and validate-fn (not max-age-ms))
      (throw (RuntimeException. "puddle: validate-fn provided but no max-age-ms given")))
    (let [allocated-ch (async/chan pool-size)
          fresh-ch (async/chan pool-size)]
      (dotimes [_ pool-size]
        (async/>!! fresh-ch (->PooledObject nil 0)))
      (let [pool (->Pool new-fn destroy-fn validate-fn max-age-ms allocated-ch fresh-ch (atom {}))]
        (when max-age-ms
          (let [; NOTE(caleb): We can make these tunables, but avoid as long as practical :)
                num-evictor-threads (max 2 (quot pool-size 100))
                evictor-delay-ms (quot max-age-ms 10)]
            (dotimes [_ num-evictor-threads]
              (safe-thread (run-evictor pool evictor-delay-ms)))))
        pool))))

(defn borrow!
  "Takes and returns a previously allocated object from the pool, if possible, or else creates a new object.
  Blocks until objects become available if the pool is at capacity and there are no available objects."
  [^Pool pool]
  (let [[^PooledObject pooled-obj _] (async/alts!! [(.allocated-ch pool) (.fresh-ch pool)]
                                                   :priority true)
        borrowed (.borrowed pool)]
    (if-let [obj (.obj pooled-obj)]
      (do
        (swap! borrowed assoc-not-existing obj pooled-obj
               "puddle: found previously-borrowed object in pool")
        obj)
      (let [obj ((.new-fn pool))
            _ (when-not obj
                (throw-str "puddle: (new-fn) yielded nil"))
            pooled-obj (assoc pooled-obj
                              :obj obj
                              :last-updated (System/currentTimeMillis))]
        (swap! borrowed assoc-not-existing obj pooled-obj
               "puddle: newly generated object already exists")
        obj))))

(defn return!
  "Returns a previously borrowed object to the pool. It is an error to return an object that was not
  previously borrowed, or to return a borrowed object more than once."
  [^Pool pool object]
  (let [borrowed (.borrowed pool)
        pooled-obj (get @borrowed object)]
    (swap! borrowed dissoc-existing object
           "puddle: returned object that was not previously borrowed or already returned")
    (async/>!! (.allocated-ch pool) (assoc pooled-obj :last-updated (System/currentTimeMillis)))))

(defn invalidate!
  "Declares an object as invalid and destroys it, freeing up capacity in the pool for a new object to be
  created. It is an error to invalidate an object that was not previously borrowed, or to invalidate a
  borrowed object more than once."
  [^Pool pool object]
  (let [borrowed (.borrowed pool)
        pooled-obj (get @borrowed object)]
    (swap! borrowed dissoc-existing object
           "puddle: invalidated object that was not previously borrowed or already returned")
    (destroy-object pool pooled-obj)))
