(ns git.internals
  (:use
   fastbeans.utils
   simplelog.use
   slingshot.slingshot)
  (:require
   [clj-jgit.porcelain :as core]
   [clj-jgit.querying :as query]
   [clj-jgit.internal :as internal]
   [clojure.java.io :as io]
   [clojure.core.memoize :as memo]
   clojure.set)
  (:import
   [org.eclipse.jgit.lib FileMode Repository ObjectIdRef ObjectId AnyObjectId Ref]
   [org.eclipse.jgit.revwalk RevWalk RevCommit RevCommitList]
   [org.eclipse.jgit.api Git LogCommand]))

(defmacro authenticated-with
  [repo-cred & body]
  `(cond
    (= (~repo-cred "type") "pubkey") (core/with-identity {:name "project-d-key"
                                                          :private (~repo-cred "private_key")
                                                          :public (~repo-cred "public_key")
                                                          :exclusive true}
                                       (let [~'url (~repo-cred "url")]
                                         ~@body))
    (= (~repo-cred "type") "password") (let [~'url (repo-url ~repo-cred)]
                                         ~@body)
    :else (throw+ (str "Unknown authentication method: " (prn-str (~repo-cred "type"))))))

(defn url-with-auth
  [^String old-url login pass]
  (let [proto (.getProtocol (java.net.URL. old-url))
        url-without-proto (.replace old-url (str proto "://") "")]
    (format "%s://%s:%s@%s" proto login pass url-without-proto)))

(defn repo-url
  [repo-cred]
  (url-with-auth (repo-cred "url") (repo-cred "login") (repo-cred "password")))

(defn get-repo
  [path]
  (core/load-repo path))

(def cached-get-repo (memo/memo-lru get-repo 100))

(defn short-commit-info
  [^Git repo ^RevWalk rev-walk ^RevCommit rev-commit]
  (let [ident (.getAuthorIdent rev-commit)
        time (-> (.getCommitTime rev-commit) (* 1000) java.util.Date.)
        message (-> (.getFullMessage rev-commit) str clojure.string/trim)]
    {:revision (.getName rev-commit)
     :author (.getName ident)
     :email (.getEmailAddress ident)
     :time time
     :message message
     :merge (> (.getParentCount rev-commit) 1)}))

(defmacro callfn
  [name args & body]
  `(defn ~name
     ~(into [] (concat ['repo-cred] args))
     (authenticated-with ~'repo-cred
                         (let [~'cache-path (~'repo-cred "path")
                               ~'repo (cached-get-repo ~'cache-path)]
                           ~@body))))

(defn refname
  [^Ref ref]
  (.getName ref))

(defn unpeel-ref
  [^Ref ref]
  [(refname ref) (.getName ^AnyObjectId (.getObjectId ^Ref ref))])

(defn unpeel-refs
  [refs]
  (into {} (map unpeel-ref refs)))

(defonce fake-repo
  (let [path (join-paths (System/getProperty "java.io.tmpdir") "fake-empty-repo.clj-jgit")]
    (if-not (.exists (io/file path))
      (core/git-init path)
      (core/load-repo path))))

(defn log-with-limit
  ([^Git repo ^Long limit]
     (let [^LogCommand cmd (.log repo)]
       (.setMaxCount cmd limit)
       (.call cmd)))
  ([^Git repo commit-ish ^Long limit]
     (let [^LogCommand cmd (.log repo)]
       (.add cmd ^AnyObjectId (internal/resolve-object commit-ish repo))
       (.setMaxCount cmd limit)
       (.call cmd))))

(defn git-fetch-mirror
  [^Git repo]
  (core/git-fetch repo "origin" "refs/heads/*:refs/heads/*"))

(defn random-access-file
  [name]
  (java.io.RandomAccessFile. (io/file name) "rw"))

(defn get-flock
  [name]
  (try
    (.. (random-access-file name) getChannel tryLock)
    (catch java.nio.channels.OverlappingFileLockException e#
      nil)))

(defn file-locked?
  [name]
  (if-let [lock (get-flock name)]
    (do (.release lock)
        false)
    true))

(defmacro with-flock
  [name & body]
  `(do
     (debug "Trying to acquire" ~name "lock")
     (if-let [lock# (get-flock ~name)]
       (try
         (debug "Lock on" ~name "acquired")
         ~@body
         (finally
           (debug "Releasing" ~name "lock")
           (.release lock#)))
       (error "Lock for" ~name "unavailable"))))

(defn lock-name
  [cache-path]
  (join-paths cache-path "fastbeans.lock"))

(defn lookup-commit
  [repo rev-walk c]
  (.lookupCommit rev-walk (internal/resolve-object c repo)))

(defn ready-flag-file
  [cache-path]
  (join-paths cache-path "fastbeans.ready"))

(defn ready!
  [cache-path]
  (touch (ready-flag-file cache-path)))
