(ns git
  (:use
   fastbeans.utils
   simplelog.use
   slingshot.slingshot
   git.internals)
  (:require
   [clj-jgit.porcelain :as core]
   [clj-jgit.querying :as query]
   [clj-jgit.internal :as internal]
   [clojure.java.io :as io]
   [clojure.core.memoize :as memo]
   clojure.set)
  (:import
   [org.eclipse.jgit.lib FileMode Repository ObjectIdRef ObjectId AnyObjectId Ref]
   [org.eclipse.jgit.revwalk RevWalk RevCommit RevCommitList]
   [org.eclipse.jgit.api Git LogCommand]))

(callfn refresh
        []
        (dofuture :refresh
                  (with-flock (lock-name cache-path)
                    (git-fetch-mirror repo))))

(callfn branches
        []
        (map refname (core/git-branch-list repo)))

(defn clone
  [repo-cred]
  (dofuture :clone
            (authenticated-with repo-cred
                                (let [cache-path (repo-cred "path")]
                                  (-> (Git/cloneRepository)
                                        (.setURI url)
                                        (.setDirectory (io/as-file cache-path))
                                        (.setBare true)
                                        (.setNoCheckout true)
                                        (.setCloneAllBranches false)
                                        (.call))
                                  (ready! cache-path)))
            (info "done with the clone")))

(callfn log
        [commit-ish limit]
        (let [rev-walk (internal/new-rev-walk repo)
              commits (log-with-limit repo commit-ish limit)]
          (->> commits
               (map (partial short-commit-info repo rev-walk)))))

(callfn rev-list
        [commit-ish limit]
        (let [commits (log-with-limit repo commit-ish limit)]
          (map #(.getName ^RevCommit %) commits)))

(callfn has-commit?
        [commit-ish]
        (if (internal/resolve-object commit-ish repo)
          true
          false))

(callfn resolve-commit
        [commit-ish]
        (if-let [^RevCommit commit (internal/resolve-object commit-ish repo)]
          (.getName commit)
          nil))

(callfn commit
        [commit-ish]
        (let [rev-walk (internal/new-rev-walk repo)
              commit (internal/bound-commit repo rev-walk (internal/resolve-object commit-ish repo))]
          (short-commit-info repo rev-walk commit)))

(callfn first-commit
        [branch-name]
        (let [^RevWalk rev-walk (internal/new-rev-walk repo)
              rev-list (RevCommitList.)]
          (.markStart rev-walk (.lookupCommit rev-walk (internal/resolve-object branch-name repo)))
          (.sort rev-walk org.eclipse.jgit.revwalk.RevSort/REVERSE)
          (doto rev-list
            (.source rev-walk)
            (.fillTo 0))
          (.getName (.get rev-list 0))))

(callfn log-range
        [c1 c2]
        (let [rev-walk (internal/new-rev-walk repo)
              commits (let [commits (core/git-log repo c1 c2)]
                        (if (empty? commits)
                          (core/git-log repo c2 c1)
                          commits))]
          (map (partial short-commit-info repo rev-walk) commits)))

(callfn commit-a-older-than-commit-b
        [c1 c2]
        (let [^RevWalk rev-walk (internal/new-rev-walk repo)
              commit-a (lookup-commit repo rev-walk c1)
              commit-b (lookup-commit repo rev-walk c2)]
          (.isMergedInto rev-walk commit-a commit-b)))

(defn ready?
  [cache-path]
  (file-exists? (ready-flag-file cache-path)))
