;; modified from https://github.com/rundis/rewrite-cljs
;; https://github.com/rundis/rewrite-cljs/blob/master/LICENSE

(ns lark.tree.parse
  (:require [lark.tree.reader :as rd]
            [lark.tree.emit :as emit]
            [lark.tree.node :as n]
            [clojure.string :as string]
    #?@(:cljs
        [[cljs.tools.reader.reader-types :as r]
         [cljs.tools.reader.edn :as edn]])
    #?@(:clj
        [[clojure.tools.reader.reader-types :as r]
         [clojure.tools.reader.edn :as edn]
         [lark.tree.util :refer [contains-identical-keyword?]]]))
  #?(:cljs (:require-macros [lark.tree.util :refer [contains-identical-keyword?]])))

#?(:cljs (enable-console-print!))

(def ^:dynamic *errors* nil)
(defn error! [info]
  (when (some? *errors*)
    (set! *errors* (conj *errors* info)))
  info)

(def ^:dynamic ^:private *delimiter* nil)
(declare parse-next)
(def non-breaking-space \u00A0)

;; identical? lookups are 10x faster than set-contains and 2x faster than js-array indexOf

(defn ^boolean newline?
  [c]
  (contains-identical-keyword? [\newline
                                \return]
                               c))

(defn ^boolean space?
  [c]
  (contains-identical-keyword? [\space
                                \tab
                                non-breaking-space]
                               c))

(defn ^boolean whitespace?
  [c]
  (or (contains-identical-keyword? [\,
                                    \space
                                    \tab
                                    non-breaking-space]
                                   c)
      (newline? c)))

(defn ^boolean boundary?
  [c]
  "Check whether a given char is a token boundary."
  (contains-identical-keyword? [\" \: \; \' \@ \^ \` \~ \( \) \[ \] \{ \} \\ nil]
                               c))

(defn ^boolean vector-contains?
  [v item]
  (let [end (count v)]
    (loop [i 0]
      (cond (= i end) false
            (identical? (v i) item) true
            :else (recur (inc i))))))

(defn read-to-boundary
  [reader allowed]
  (rd/read-until
    reader
    #(and (or (whitespace? %)
              (boundary? %))
          (not (allowed %)))))

(defn read-to-char-boundary
  [reader]
  (let [c (r/read-char reader)]
    (str c
         (if ^boolean (not (identical? c \\))
           (read-to-boundary reader #{})
           ""))))

(defn dispatch
  [c]
  (cond (identical? c *delimiter*) :matched-delimiter
        (nil? c) :eof

        (identical? c \,) :comma

        (or (identical? c \space)
            (identical? c non-breaking-space)
            (identical? c \tab)) :space

        (newline? c) :newline

        (identical? c \^) :meta
        (identical? c \#) :sharp
        (identical? c \() :list
        (identical? c \[) :vector
        (identical? c \{) :map

        (or (identical? c \})
            (identical? c \])
            (identical? c \))) :unmatched-delimiter

        (identical? c \~) :unquote
        (identical? c \') :quote
        (identical? c \`) :syntax-quote
        (identical? c \;) :comment
        (identical? c \@) :deref
        (identical? c \") :string
        (identical? c \:) :keyword
        :else :token))

(defn parse-delim
  [reader delimiter]
  (r/read-char reader)
  (rd/read-repeatedly reader #(binding [*delimiter* delimiter]
                                (parse-next %))))

(defn ^boolean printable-only? [n]
  (contains? #{:space :comma :newline :comment :comment-block}
             (:tag n)))

(defn parse-printables
  [reader node-tag n & [ignore?]]
  (when-not (nil? ignore?)
    (r/read-char reader))
  (rd/read-n
    reader
    node-tag
    parse-next
    (complement printable-only?)
    n))

(def brackets {\( \)
               \[ \]
               \{ \}})

(defn parse-token
  "Parse a single token."
  [reader]
  (let [first-char (r/read-char reader)
        s          (->> (if ^boolean (identical? first-char \\)
                          (read-to-char-boundary reader)
                          (read-to-boundary reader #{}))
                        (str first-char))
        ;; determine if string is a symbol, inferring 'yes' on a
        ;; symbol-related read error:
        sexp       (try (edn/read-string s)
                        (catch #?(:cljs js/Error
                                  :clj  Exception) e
                          (let [message #?(:cljs (ex-message e)
                                           :clj (.getMessage e))]
                            ;; TODO
                            ;; a better way to determine that this is an invalid symbol?
                            (when (string/ends-with? message "/.")
                              ::invalid-symbol))))
        is-symbol  ^boolean (or (symbol? sexp) (= sexp ::invalid-symbol))]
    (if is-symbol
      [:symbol (str s (read-to-boundary reader #{\' \:}))]
      [:token s])))

(defn parse-keyword
  [reader]
  (r/read-char reader)
  (if-let [c (r/peek-char reader)]
    (if ^boolean (identical? c \:)
      [:namespaced-keyword (edn/read reader)]
      (do (r/unread reader \:)
          ;; TODO
          ;; handle invalid keywords
          [:keyword (edn/read reader)]))
    (rd/throw-reader reader "unexpected EOF while reading keyword.")))

(defn parse-sharp
  [reader]
  (r/read-char reader)
  (case (r/peek-char reader)
    nil (rd/throw-reader reader "Unexpected EOF.")
    \{ [:set (parse-delim reader \})]
    \( [:fn (parse-delim reader \))]
    \" [:regex (rd/read-string-data reader)]
    \^ [:reader-meta (parse-printables reader :reader-meta 2 true)]
    \' [:var (parse-printables reader :var 1 true)]
    \_ [:uneval (parse-printables reader :uneval 1 true)]
    \? (do
         (r/read-char reader)
         (let [read-next #(parse-printables reader :reader-macro 1)
               opts      (case (r/peek-char reader)
                           \( {:prefix  "#?"
                               :splice? true}
                           \@ (do (r/read-char reader)
                                  {:prefix  "#?@"
                                   :splice? true})
                           ;; no idea what this would be, but its \? prefixed
                           (do (rd/unread reader \?)
                               {:prefix (str "#?" (read-next))}))
               value     (read-next)]
           [:reader-conditional value opts]))
    [:reader-macro (parse-printables reader :reader-macro 2)]))

(defn parse-unquote
  [reader]
  (r/read-char reader)
  (let [c (r/peek-char reader)]
    (if ^boolean (identical? c \@)
      [:unquote-splicing (parse-printables reader :unquote 1 true)]
      [:unquote (parse-printables reader :unquote 1)])))

#_(defn parse-comment-block [reader opening-newline?]
    [:comment-block (loop [text (if opening-newline? \newline "")]
                      (rd/read-while reader #{\;})
                      (when (space? (r/peek-char reader))
                        (r/read-char reader))

                      (let [comment-line    (rd/read-until reader #(or (nil? %)
                                                                       (identical? % \newline)
                                                                       (identical? % \return)))
                            next-whitespace (loop [whitespace nil]
                                              (let [next-char (r/peek-char reader)]
                                                (if (whitespace? next-char)
                                                  (do (r/read-char reader)
                                                      (recur (str whitespace next-char)))
                                                  whitespace)))
                            next-comment?   (and next-whitespace
                                                 (not (space? (last next-whitespace)))
                                                 (= \; (r/peek-char reader)))]
                        (if next-comment?
                          (recur (str text comment-line next-whitespace))
                          (do
                            (when next-whitespace
                              (doseq [c (reverse next-whitespace)]
                                (r/unread reader c)))
                            (str text comment-line)))))])

(defn parse-next*
  [reader]
  (let [c   (r/peek-char reader)
        tag (dispatch c)]
    (case tag
      :token (parse-token reader)
      :keyword (parse-keyword reader)
      :sharp (parse-sharp reader)
      :comment (do (rd/ignore reader)
                   [tag (let [content (rd/read-until reader (fn [x] (or (nil? x) (#{\newline \return} x))))]
                          content
                          #_(cond-> content
                                    (newline? (r/peek-char reader)) (str (r/read-char reader))))])
      (:deref
        :quote
        :syntax-quote) [tag (parse-printables reader tag 1 true)]

      :unquote (parse-unquote reader)

      :newline (do (rd/ignore reader)
                   [tag (str "\n" (rd/read-while reader space?))])

      :comma [tag (rd/read-while reader #(identical? % c))]
      :space [tag (rd/read-while reader space?)]
      (:list
        :vector
        :map) [tag (parse-delim reader (get brackets c))]

      :matched-delimiter (do (r/read-char reader) nil)
      (:eof :unmatched-delimiter) (let [the-error (error! [(keyword "error" (name tag)) (let [[line col] (rd/position reader)]
                                                                                          {:position  {:line       line
                                                                                                       :column     col
                                                                                                       :end-line   line
                                                                                                       :end-column (inc col)}
                                                                                           :delimiter *delimiter*})])]
                                    (r/read-char reader)
                                    the-error)
      :meta (do (r/read-char reader)
                [tag (parse-printables reader :meta 2)])
      :string [tag (rd/read-string-data reader)])))

(defn parse-next
  [reader]
  (rd/read-with-position reader parse-next*))

(defn indexing-reader
  "Create reader for strings."
  [s]
  (r/indexing-push-back-reader
    (r/string-push-back-reader s 500)))

(defn comment-block-child? [{:keys [tag]}]
  (contains-identical-keyword? [:space :newline :comment] tag))

(defn ast*
  [s]
  (binding [*errors* []]
    (loop [reader (indexing-reader s)
           source ""
           values []]
      (if-some [next-thing (rd/read-with-position reader parse-next*)]
        (let [next-source (emit/string next-thing)]
          (recur reader
                 (str source next-source)
                 (conj values (assoc next-thing :source next-source))))
        {:value      values
         :source     source
         :tag        :base
         :errors     *errors*
         :line       0
         :column     0
         :end-line   (r/get-line-number reader)
         :end-column (r/get-column-number reader)}))))

(defn ast
  "Parse ClojureScript source code to AST"
  ([source] (ast nil source))
  ([ns source]
   (let [{out-str :source :as the-ast} (ast* source)
         modified-source? (not= source out-str)
         result           (assoc (if modified-source?
                                   (ast* out-str)
                                   the-ast) :string out-str
                                            :modified-source? modified-source?)]
     result)))

(defn normalize-comment-line [s]
  (string/replace s #"^;+\s?" ""))



(comment

  ;; IN PROGRESS
  ;; thinking about a better way to group comment and code blocks
  ;; ...contemplating a transducer, or similar thing?

  (defn conj-while [[out in] xform]
    (loop [out out
           in  in]
      (if-let [form (xform (peek in))]
        (recur (update-in out [(dec (count out)) :value] conj form)
               (subvec in 1))
        [out in])))

  (groups {:comment-block {:init {:tag   :comment-block
                                  :value ""}
                           :pred comment-block-child?
                           :conj (fn [oldval node]
                                   (str oldval (-> (emit/string node)
                                                   (normalize-comment-line))))}
           :code-block    {:init {:tag   :base
                                  :value []}
                           :pred (complement comment-block-child?)
                           :conj (fn [oldval node]
                                   (conj oldval node))}} nodes))

(defn group-comment-blocks
  "Put consecutive top-level whitespace and comment nodes into :comment-blocks"
  [ast]
  (update ast :value (fn [nodes]
                       (reduce (fn [out node]
                                 (if (comment-block-child? node)
                                   (if (= :comment-block (:tag (peek out)))
                                     (update-in out [(dec (count out)) :value] str (-> (emit/string node)
                                                                                       (normalize-comment-line)))
                                     (conj out (merge node
                                                      {:tag   :comment-block
                                                       :value (normalize-comment-line (emit/string node))})))
                                   (conj out node))) [] nodes))))

#_(defn flatten-comment-blocks [ast]
    (update ast :value (fn [nodes]
                         (mapv (fn [{:keys [value tag] :as node}]
                                 (if (= :comment-block tag)
                                   (assoc node :value (string/join (mapv emit/string value)))
                                   node)) nodes))))

(defn shape [{:keys [tag value] :as node}]
  (if (= tag :base)
    (mapv shape value)
    (if (n/may-contain-children? node)
      [tag (mapv shape value)]
      tag)))

(prn :parse-core)
(time (ast ";   Copyright (c) Rich Hickey. All rights reserved.\n;   The use and distribution terms for this software are covered by the\n;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)\n;   which can be found in the file epl-v10.html at the root of this distribution.\n;   By using this software in any fashion, you are agreeing to be bound by\n;   the terms of this license.\n;   You must not remove this notice, or any other, from this software.\n\n(ns cljs.core\n  (:refer-clojure :exclude [-> ->> .. amap and areduce alength aclone assert assert-args binding bound-fn case comment\n                            cond condp declare definline definterface defmethod defmulti defn defn- defonce\n                            defprotocol defrecord defstruct deftype delay destructure doseq dosync dotimes doto\n                            extend-protocol extend-type fn for future gen-class gen-interface\n                            if-let if-not import io! lazy-cat lazy-seq let letfn locking loop\n                            memfn ns or proxy proxy-super pvalues refer-clojure reify sync time\n                            when when-first when-let when-not while with-bindings with-in-str\n                            with-loading-context with-local-vars with-open with-out-str with-precision with-redefs\n                            satisfies? identical? true? false? number? nil? instance? symbol? keyword? string? str get\n                            make-array vector list hash-map array-map hash-set\n\n                            aget aset\n                            + - * / < <= > >= == zero? pos? neg? inc dec max min mod\n                            byte char short int long float double\n                            unchecked-byte unchecked-char unchecked-short unchecked-int\n                            unchecked-long unchecked-float unchecked-double\n                            unchecked-add unchecked-add-int unchecked-dec unchecked-dec-int\n                            unchecked-divide unchecked-divide-int unchecked-inc unchecked-inc-int\n                            unchecked-multiply unchecked-multiply-int unchecked-negate unchecked-negate-int\n                            unchecked-subtract unchecked-subtract-int unchecked-remainder-int\n                            unsigned-bit-shift-right\n\n                            bit-and bit-and-not bit-clear bit-flip bit-not bit-or bit-set\n                            bit-test bit-shift-left bit-shift-right bit-xor defmacro\n\n                            cond-> cond->> as-> some-> some->>\n\n                            require use refer-clojure\n\n                            if-some when-some test ns-interns ns-unmap var vswap! macroexpand-1 macroexpand\n                            some? resolve\n                            #?@(:cljs [alias coercive-not coercive-not= coercive-= coercive-boolean\n                                       truth_ js-arguments js-delete js-in js-debugger exists? divide js-mod\n                                       unsafe-bit-and bit-shift-right-zero-fill mask bitpos caching-hash\n                                       defcurried rfn specify! js-this this-as implements? array js-obj\n                                       simple-benchmark gen-apply-to js-str es6-iterable load-file* undefined?\n                                       specify copy-arguments goog-define js-comment js-inline-comment\n                                       unsafe-cast require-macros use-macros gen-apply-to-simple])])\n  #?(:cljs (:require-macros [cljs.core :as core]\n                            [cljs.support :refer [assert-args]]))\n  (:require clojure.walk\n            clojure.set\n            [clojure.string :as string]\n            [cljs.compiler :as comp]\n            [cljs.env :as env]\n            #?(:clj [cljs.support :refer [assert-args]])\n            #?(:cljs [cljs.core :as core])\n            #?(:cljs [cljs.analyzer :as ana])))\n\n#?(:clj (alias 'core 'clojure.core))\n#?(:clj (alias 'ana 'cljs.analyzer))\n\n#?(:clj\n   (core/defmacro import-macros [ns [& vars]]\n     (core/let [ns (find-ns ns)\n                vars (map #(ns-resolve ns %) vars)\n                syms (map\n                       (core/fn [^clojure.lang.Var v]\n                         (core/-> v .sym\n                           (with-meta\n                             (merge\n                               {:macro true}\n                               (update-in (select-keys (meta v) [:arglists :doc :file :line])\n                                 [:arglists] (core/fn [arglists] `(quote ~arglists)))))))\n                       vars)\n                defs (map\n                       (core/fn [sym var]\n                         (core/let [{:keys [arglists doc file line]} (meta sym)]\n                           `(do\n                              (def ~sym (deref ~var))\n                              ;for AOT compilation\n                              (alter-meta! (var ~sym) assoc\n                                :macro true\n                                :arglists ~arglists\n                                :doc ~doc\n                                :file ~file\n                                :line ~line))))\n                       syms vars)]\n       `(do ~@defs\n            :imported))))\n\n#?(:clj\n   (import-macros clojure.core\n     [-> ->> .. assert comment cond\n      declare defn-\n      doto\n      extend-protocol fn for\n      if-let if-not letfn\n      memfn\n      when when-first when-let when-not while\n      cond-> cond->> as-> some-> some->>\n      if-some when-some]))\n\n#?(:cljs\n   (core/defmacro ->\n     \"Threads the expr through the forms. Inserts x as the\n     second item in the first form, making a list of it if it is not a\n     list already. If there are more forms, inserts the first form as the\n     second item in second form, etc.\"\n     [x & forms]\n     (core/loop [x x, forms forms]\n       (if forms\n         (core/let [form (first forms)\n                    threaded (if (seq? form)\n                               (with-meta `(~(first form) ~x ~@(next form)) (meta form))\n                               (core/list form x))]\n           (recur threaded (next forms)))\n         x))))\n\n#?(:cljs\n   (core/defmacro ->>\n     \"Threads the expr through the forms. Inserts x as the\n     last item in the first form, making a list of it if it is not a\n     list already. If there are more forms, inserts the first form as the\n     last item in second form, etc.\"\n     [x & forms]\n     (core/loop [x x, forms forms]\n       (if forms\n         (core/let [form (first forms)\n                    threaded (if (seq? form)\n                               (with-meta `(~(first form) ~@(next form) ~x) (meta form))\n                               (core/list form x))]\n           (recur threaded (next forms)))\n         x))))\n\n#?(:cljs\n   (core/defmacro ..\n     \"form => fieldName-symbol or (instanceMethodName-symbol args*)\n\n     Expands into a member access (.) of the first member on the first\n     argument, followed by the next member on the result, etc. For\n     instance:\n\n     (.. System (getProperties) (get \\\"os.name\\\"))\n\n     expands to:\n\n     (. (. System (getProperties)) (get \\\"os.name\\\"))\n\n     but is easier to write, read, and understand.\"\n     ([x form] `(. ~x ~form))\n     ([x form & more] `(.. (. ~x ~form) ~@more))))\n\n#?(:cljs\n   (core/defmacro comment\n     \"Ignores body, yields nil\"\n     [& body]))\n\n#?(:cljs\n   (core/defmacro cond\n     \"Takes a set of test/expr pairs. It evaluates each test one at a\n     time.  If a test returns logical true, cond evaluates and returns\n     the value of the corresponding expr and doesn't evaluate any of the\n     other tests or exprs. (cond) returns nil.\"\n     {:added \"1.0\"}\n     [& clauses]\n     (core/when clauses\n       (core/list 'if (first clauses)\n         (if (next clauses)\n           (second clauses)\n           (throw (js/Error. \"cond requires an even number of forms\")))\n         (cons 'cljs.core/cond (next (next clauses)))))))\n\n#?(:cljs\n   (core/defmacro declare\n     \"defs the supplied var names with no bindings, useful for making forward declarations.\"\n     [& names] `(do ~@(map #(core/list 'def (vary-meta % assoc :declared true)) names))))\n\n#?(:cljs\n   (core/defmacro doto\n     \"Evaluates x then calls all of the methods and functions with the\n     value of x supplied at the front of the given arguments.  The forms\n     are evaluated in order.  Returns x.\n\n     (doto (new java.util.HashMap) (.put \\\"a\\\" 1) (.put \\\"b\\\" 2))\"\n     [x & forms]\n     (core/let [gx (gensym)]\n       `(let [~gx ~x]\n          ~@(map (core/fn [f]\n                   (if (seq? f)\n                     `(~(first f) ~gx ~@(next f))\n                     `(~f ~gx)))\n              forms)\n          ~gx))))\n\n#?(:cljs\n   (core/defn- parse-impls [specs]\n     (core/loop [ret {} s specs]\n       (if (seq s)\n         (recur (assoc ret (first s) (take-while seq? (next s)))\n           (drop-while seq? (next s)))\n         ret))))\n\n#?(:cljs\n   (core/defn- emit-extend-protocol [p specs]\n     (core/let [impls (parse-impls specs)]\n       `(do\n          ~@(map (core/fn [[t fs]]\n                   `(extend-type ~t ~p ~@fs))\n              impls)))))\n\n#?(:cljs\n   (core/defmacro extend-protocol\n     \"Useful when you want to provide several implementations of the same\n     protocol all at once. Takes a single protocol and the implementation\n     of that protocol for one or more types. Expands into calls to\n     extend-type:\n\n     (extend-protocol Protocol\n       AType\n         (foo [x] ...)\n         (bar [x y] ...)\n       BType\n         (foo [x] ...)\n         (bar [x y] ...)\n       AClass\n         (foo [x] ...)\n         (bar [x y] ...)\n       nil\n         (foo [x] ...)\n         (bar [x y] ...))\n\n     expands into:\n\n     (do\n      (clojure.core/extend-type AType Protocol\n        (foo [x] ...)\n        (bar [x y] ...))\n      (clojure.core/extend-type BType Protocol\n        (foo [x] ...)\n        (bar [x y] ...))\n      (clojure.core/extend-type AClass Protocol\n        (foo [x] ...)\n        (bar [x y] ...))\n      (clojure.core/extend-type nil Protocol\n        (foo [x] ...)\n        (bar [x y] ...)))\"\n     [p & specs]\n     (emit-extend-protocol p specs)))\n\n#?(:cljs\n   (core/defn ^{:private true}\n   maybe-destructured\n     [params body]\n     (if (every? core/symbol? params)\n       (cons params body)\n       (core/loop [params params\n                   new-params (with-meta [] (meta params))\n                   lets []]\n         (if params\n           (if (core/symbol? (first params))\n             (recur (next params) (conj new-params (first params)) lets)\n             (core/let [gparam (gensym \"p__\")]\n               (recur (next params) (conj new-params gparam)\n                 (core/-> lets (conj (first params)) (conj gparam)))))\n           `(~new-params\n              (let ~lets\n                ~@body)))))))\n\n#?(:cljs\n   (core/defmacro fn\n     \"params => positional-params* , or positional-params* & next-param\n     positional-param => binding-form\n     next-param => binding-form\n     name => symbol\n\n     Defines a function\"\n     {:forms '[(fn name? [params*] exprs*) (fn name? ([params*] exprs*) +)]}\n     [& sigs]\n     (core/let [name (if (core/symbol? (first sigs)) (first sigs) nil)\n                sigs (if name (next sigs) sigs)\n                sigs (if (vector? (first sigs))\n                       (core/list sigs)\n                       (if (seq? (first sigs))\n                         sigs\n                         ;; Assume single arity syntax\n                         (throw (js/Error.\n                                  (if (seq sigs)\n                                    (core/str \"Parameter declaration \"\n                                      (core/first sigs)\n                                      \" should be a vector\")\n                                    (core/str \"Parameter declaration missing\"))))))\n                psig (fn* [sig]\n                       ;; Ensure correct type before destructuring sig\n                       (core/when (not (seq? sig))\n                         (throw (js/Error.\n                                  (core/str \"Invalid signature \" sig\n                                    \" should be a list\"))))\n                       (core/let [[params & body] sig\n                                  _ (core/when (not (vector? params))\n                                      (throw (js/Error.\n                                               (if (seq? (first sigs))\n                                                 (core/str \"Parameter declaration \" params\n                                                   \" should be a vector\")\n                                                 (core/str \"Invalid signature \" sig\n                                                   \" should be a list\")))))\n                                  conds (core/when (core/and (next body) (map? (first body)))\n                                          (first body))\n                                  body (if conds (next body) body)\n                                  conds (core/or conds (meta params))\n                                  pre (:pre conds)\n                                  post (:post conds)\n                                  body (if post\n                                         `((let [~'% ~(if (core/< 1 (count body))\n                                                        `(do ~@body)\n                                                        (first body))]\n                                             ~@(map (fn* [c] `(assert ~c)) post)\n                                             ~'%))\n                                         body)\n                                  body (if pre\n                                         (concat (map (fn* [c] `(assert ~c)) pre)\n                                           body)\n                                         body)]\n                         (maybe-destructured params body)))\n                new-sigs (map psig sigs)]\n       (with-meta\n         (if name\n           (list* 'fn* name new-sigs)\n           (cons 'fn* new-sigs))\n         (meta &form)))))\n\n#?(:cljs\n   (core/defmacro defn-\n     \"same as defn, yielding non-public def\"\n     [name & decls]\n     (list* `defn (with-meta name (assoc (meta name) :private true)) decls)))\n\n#?(:cljs\n   (core/defmacro if-let\n     \"bindings => binding-form test\n\n     If test is true, evaluates then with binding-form bound to the value of\n     test, if not, yields else\"\n     ([bindings then]\n      `(if-let ~bindings ~then nil))\n     ([bindings then else & oldform]\n      (assert-args if-let\n        (vector? bindings) \"a vector for its binding\"\n        (empty? oldform) \"1 or 2 forms after binding vector\"\n        (= 2 (count bindings)) \"exactly 2 forms in binding vector\")\n      (core/let [form (bindings 0) tst (bindings 1)]\n        `(let [temp# ~tst]\n           (if temp#\n             (let [~form temp#]\n               ~then)\n             ~else))))))\n\n#?(:cljs\n   (core/defmacro if-not\n     \"Evaluates test. If logical false, evaluates and returns then expr,\n     otherwise else expr, if supplied, else nil.\"\n     ([test then] `(if-not ~test ~then nil))\n     ([test then else]\n      `(if (not ~test) ~then ~else))))\n\n#?(:cljs\n   (core/defmacro letfn\n     \"fnspec ==> (fname [params*] exprs) or (fname ([params*] exprs)+)\n\n     Takes a vector of function specs and a body, and generates a set of\n     bindings of functions to their names. All of the names are available\n     in all of the definitions of the functions, as well as the body.\"\n     {:forms '[(letfn [fnspecs*] exprs*)],\n      :special-form true, :url nil}\n     [fnspecs & body]\n     `(letfn* ~(vec (interleave (map first fnspecs)\n                      (map #(cons `fn %) fnspecs)))\n        ~@body)))\n\n#?(:cljs\n   (core/defmacro memfn\n     \"Expands into code that creates a fn that expects to be passed an\n     object and any args and calls the named instance method on the\n     object passing the args. Use when you want to treat a Java method as\n     a first-class fn. name may be type-hinted with the method receiver's\n     type in order to avoid reflective calls.\"\n     [name & args]\n     (core/let [t (with-meta (gensym \"target\")\n               (meta name))]\n       `(fn [~t ~@args]\n          (. ~t (~name ~@args))))))\n\n#?(:cljs\n   (core/defmacro when\n     \"Evaluates test. If logical true, evaluates body in an implicit do.\"\n     [test & body]\n     (core/list 'if test (cons 'do body))))\n\n#?(:cljs\n   (core/defmacro when-first\n     \"bindings => x xs\n\n     Roughly the same as (when (seq xs) (let [x (first xs)] body)) but xs is evaluated only once\"\n     [bindings & body]\n     (assert-args when-first\n       (vector? bindings) \"a vector for its binding\"\n       (= 2 (count bindings)) \"exactly 2 forms in binding vector\")\n     (core/let [[x xs] bindings]\n       `(when-let [xs# (seq ~xs)]\n          (let [~x (first xs#)]\n            ~@body)))))\n\n#?(:cljs\n   (core/defmacro when-let\n     \"bindings => binding-form test\n\n     When test is true, evaluates body with binding-form bound to the value of test\"\n     [bindings & body]\n     (assert-args when-let\n       (vector? bindings) \"a vector for its binding\"\n       (= 2 (count bindings)) \"exactly 2 forms in binding vector\")\n     (core/let [form (bindings 0) tst (bindings 1)]\n       `(let [temp# ~tst]\n          (when temp#\n            (let [~form temp#]\n              ~@body))))))\n\n#?(:cljs\n   (core/defmacro when-not\n     \"Evaluates test. If logical false, evaluates body in an implicit do.\"\n     [test & body]\n     (core/list 'if test nil (cons 'do body))))\n\n#?(:cljs\n   (core/defmacro while\n     \"Repeatedly executes body while test expression is true. Presumes\n     some side-effect will cause test to become false/nil. Returns nil\"\n     [test & body]\n     `(loop []\n        (when ~test\n          ~@body\n          (recur)))))\n\n#?(:cljs\n   (core/defmacro cond->\n     \"Takes an expression and a set of test/form pairs. Threads expr (via ->)\n     through each form for which the corresponding test\n     expression is true. Note that, unlike cond branching, cond-> threading does\n     not short circuit after the first true test expression.\"\n     [expr & clauses]\n     (core/assert (even? (count clauses)))\n     (core/let [g (gensym)\n                steps (map (core/fn [[test step]] `(if ~test (-> ~g ~step) ~g))\n                        (partition 2 clauses))]\n       `(let [~g ~expr\n              ~@(interleave (repeat g) (butlast steps))]\n          ~(if (empty? steps)\n             g\n             (last steps))))))\n\n#?(:cljs\n   (core/defmacro cond->>\n     \"Takes an expression and a set of test/form pairs. Threads expr (via ->>)\n     through each form for which the corresponding test expression\n     is true.  Note that, unlike cond branching, cond->> threading does not short circuit\n     after the first true test expression.\"\n     [expr & clauses]\n     (core/assert (even? (count clauses)))\n     (core/let [g (gensym)\n                steps (map (core/fn [[test step]] `(if ~test (->> ~g ~step) ~g))\n                        (partition 2 clauses))]\n       `(let [~g ~expr\n              ~@(interleave (repeat g) (butlast steps))]\n          ~(if (empty? steps)\n             g\n             (last steps))))))\n\n#?(:cljs\n   (core/defmacro as->\n     \"Binds name to expr, evaluates the first form in the lexical context\n     of that binding, then binds name to that result, repeating for each\n     successive form, returning the result of the last form.\"\n     [expr name & forms]\n     `(let [~name ~expr\n            ~@(interleave (repeat name) (butlast forms))]\n        ~(if (empty? forms)\n           name\n           (last forms)))))\n\n#?(:cljs\n   (core/defmacro some->\n     \"When expr is not nil, threads it into the first form (via ->),\n     and when that result is not nil, through the next etc\"\n     [expr & forms]\n     (core/let [g (gensym)\n                steps (map (core/fn [step] `(if (nil? ~g) nil (-> ~g ~step)))\n                        forms)]\n       `(let [~g ~expr\n              ~@(interleave (repeat g) (butlast steps))]\n          ~(if (empty? steps)\n             g\n             (last steps))))))\n\n#?(:cljs\n   (core/defmacro some->>\n     \"When expr is not nil, threads it into the first form (via ->>),\n     and when that result is not nil, through the next etc\"\n     [expr & forms]\n     (core/let [g (gensym)\n                steps (map (core/fn [step] `(if (nil? ~g) nil (->> ~g ~step)))\n                        forms)]\n       `(let [~g ~expr\n              ~@(interleave (repeat g) (butlast steps))]\n          ~(if (empty? steps)\n             g\n             (last steps))))))\n\n#?(:cljs\n   (core/defmacro if-some\n     \"bindings => binding-form test\n\n      If test is not nil, evaluates then with binding-form bound to the\n      value of test, if not, yields else\"\n     ([bindings then]\n      `(if-some ~bindings ~then nil))\n     ([bindings then else & oldform]\n      (assert-args if-some\n        (vector? bindings) \"a vector for its binding\"\n        (empty? oldform) \"1 or 2 forms after binding vector\"\n        (= 2 (count bindings)) \"exactly 2 forms in binding vector\")\n      (core/let [form (bindings 0) tst (bindings 1)]\n        `(let [temp# ~tst]\n           (if (nil? temp#)\n             ~else\n             (let [~form temp#]\n               ~then)))))))\n\n#?(:cljs\n   (core/defmacro when-some\n     \"bindings => binding-form test\n\n      When test is not nil, evaluates body with binding-form bound to the\n      value of test\"\n     [bindings & body]\n     (assert-args when-some\n       (vector? bindings) \"a vector for its binding\"\n       (= 2 (count bindings)) \"exactly 2 forms in binding vector\")\n     (core/let [form (bindings 0) tst (bindings 1)]\n       `(let [temp# ~tst]\n          (if (nil? temp#)\n            nil\n            (let [~form temp#]\n              ~@body))))))\n\n(core/defn- ^{:dynamic true} assert-valid-fdecl\n  \"A good fdecl looks like (([a] ...) ([a b] ...)) near the end of defn.\"\n  [fdecl]\n  (core/when (empty? fdecl)\n    (throw\n      #?(:clj  (IllegalArgumentException. \"Parameter declaration missing\")\n         :cljs (js/Error. \"Parameter declaration missing\"))))\n  (core/let [argdecls\n             (map\n               #(if (seq? %)\n                 (first %)\n                 (throw\n                   #?(:clj (IllegalArgumentException.\n                             (if (seq? (first fdecl))\n                               (core/str \"Invalid signature \\\"\"\n                                 %\n                                 \"\\\" should be a list\")\n                               (core/str \"Parameter declaration \\\"\"\n                                 %\n                                 \"\\\" should be a vector\")))\n                      :cljs (js/Error.\n                              (if (seq? (first fdecl))\n                                (core/str \"Invalid signature \\\"\"\n                                  %\n                                  \"\\\" should be a list\")\n                                (core/str \"Parameter declaration \\\"\"\n                                  %\n                                  \"\\\" should be a vector\"))))))\n               fdecl)\n             bad-args (seq (remove #(vector? %) argdecls))]\n    (core/when bad-args\n      (throw\n        #?(:clj (IllegalArgumentException.\n                  (core/str \"Parameter declaration \\\"\" (first bad-args)\n                    \"\\\" should be a vector\"))\n           :cljs (js/Error.\n                   (core/str \"Parameter declaration \\\"\" (first bad-args)\n                     \"\\\" should be a vector\")))))))\n\n(def\n  ^{:private true}\n  sigs\n  (core/fn [fdecl]\n    (assert-valid-fdecl fdecl)\n    (core/let [asig\n               (core/fn [fdecl]\n                 (core/let [arglist (first fdecl)\n                            ;elide implicit macro args\n                            arglist (if #?(:clj (clojure.lang.Util/equals '&form (first arglist))\n                                           :cljs (= '&form (first arglist)))\n                                      #?(:clj (clojure.lang.RT/subvec arglist 2 (clojure.lang.RT/count arglist))\n                                         :cljs (subvec arglist 2 (count arglist)))\n                                      arglist)\n                            body (next fdecl)]\n                   (if (map? (first body))\n                     (if (next body)\n                       (with-meta arglist (conj (if (meta arglist) (meta arglist) {}) (first body)))\n                       arglist)\n                     arglist)))]\n      (if (seq? (first fdecl))\n        (core/loop [ret [] fdecls fdecl]\n          (if fdecls\n            (recur (conj ret (asig (first fdecls))) (next fdecls))\n            (seq ret)))\n        (core/list (asig fdecl))))))\n\n(core/defmacro defonce\n  \"defs name to have the root value of init iff the named var has no root value,\n  else init is unevaluated\"\n  [x init]\n  `(when-not (exists? ~x)\n     (def ~x ~init)))\n\n(core/defn destructure [bindings]\n  (core/let [bents (partition 2 bindings)\n             pb (core/fn pb [bvec b v]\n                  (core/let [pvec\n                             (core/fn [bvec b val]\n                               (core/let [gvec (gensym \"vec__\")\n                                          gseq (gensym \"seq__\")\n                                          gfirst (gensym \"first__\")\n                                          has-rest (some #{'&} b)]\n                                 (core/loop [ret (core/let [ret (conj bvec gvec val)]\n                                                   (if has-rest\n                                                     (conj ret gseq (core/list `seq gvec))\n                                                     ret))\n                                             n 0\n                                             bs b\n                                             seen-rest? false]\n                                   (if (seq bs)\n                                     (core/let [firstb (first bs)]\n                                       (core/cond\n                                         (= firstb '&) (recur (pb ret (second bs) gseq)\n                                                              n\n                                                              (nnext bs)\n                                                              true)\n                                         (= firstb :as) (pb ret (second bs) gvec)\n                                         :else (if seen-rest?\n                                                 (throw #?(:clj (new Exception \"Unsupported binding form, only :as can follow & parameter\")\n                                                           :cljs (new js/Error \"Unsupported binding form, only :as can follow & parameter\")))\n                                                 (recur (pb (if has-rest\n                                                              (conj ret\n                                                                    gfirst `(first ~gseq)\n                                                                    gseq `(next ~gseq))\n                                                              ret)\n                                                            firstb\n                                                            (if has-rest\n                                                              gfirst\n                                                              (core/list `nth gvec n nil)))\n                                                        (core/inc n)\n                                                        (next bs)\n                                                        seen-rest?))))\n                                     ret))))\n                             pmap\n                             (core/fn [bvec b v]\n                               (core/let [gmap (gensym \"map__\")\n                                          defaults (:or b)]\n                                 (core/loop [ret (core/-> bvec (conj gmap) (conj v)\n                                                          (conj gmap) (conj `(if (implements? ISeq ~gmap) (apply cljs.core/hash-map ~gmap) ~gmap))\n                                                     ((core/fn [ret]\n                                                        (if (:as b)\n                                                          (conj ret (:as b) gmap)\n                                                          ret))))\n                                             bes (core/let [transforms\n                                                            (reduce\n                                                              (core/fn [transforms mk]\n                                                                (if (core/keyword? mk)\n                                                                  (core/let [mkns (namespace mk)\n                                                                        mkn (name mk)]\n                                                                    (core/cond (= mkn \"keys\") (assoc transforms mk #(keyword (core/or mkns (namespace %)) (name %)))\n                                                                               (= mkn \"syms\") (assoc transforms mk #(core/list `quote (symbol (core/or mkns (namespace %)) (name %))))\n                                                                               (= mkn \"strs\") (assoc transforms mk core/str)\n                                                                               :else transforms))\n                                                                  transforms))\n                                                              {}\n                                                              (keys b))]\n                                                   (reduce\n                                                     (core/fn [bes entry]\n                                                       (reduce #(assoc %1 %2 ((val entry) %2))\n                                                         (dissoc bes (key entry))\n                                                         ((key entry) bes)))\n                                                     (dissoc b :as :or)\n                                                     transforms))]\n                                   (if (seq bes)\n                                     (core/let [bb (key (first bes))\n                                                bk (val (first bes))\n                                                local (if #?(:clj  (core/instance? clojure.lang.Named bb)\n                                                             :cljs (cljs.core/implements? INamed bb))\n                                                          (with-meta (symbol nil (name bb)) (meta bb))\n                                                        bb)\n                                                bv (if (contains? defaults local)\n                                                     (core/list 'cljs.core/get gmap bk (defaults local))\n                                                     (core/list 'cljs.core/get gmap bk))]\n                                       (recur\n                                         (if (core/or (core/keyword? bb) (core/symbol? bb)) ;(ident? bb)\n                                           (core/-> ret (conj local bv))\n                                           (pb ret bb bv))\n                                              (next bes)))\n                                     ret))))]\n                    (core/cond\n                      (core/symbol? b) (core/-> bvec (conj (if (namespace b) (symbol (name b)) b)) (conj v))\n                      (core/keyword? b) (core/-> bvec (conj (symbol (name b))) (conj v))\n                      (vector? b) (pvec bvec b v)\n                      (map? b) (pmap bvec b v)\n                      :else (throw\n                             #?(:clj (new Exception (core/str \"Unsupported binding form: \" b))\n                                :cljs (new js/Error (core/str \"Unsupported binding form: \" b)))))))\n             process-entry (core/fn [bvec b] (pb bvec (first b) (second b)))]\n    (if (every? core/symbol? (map first bents))\n      bindings\n      (core/if-let [kwbs (seq (filter #(core/keyword? (first %)) bents))]\n        (throw\n          #?(:clj (new Exception (core/str \"Unsupported binding key: \" (ffirst kwbs)))\n             :cljs (new js/Error (core/str \"Unsupported binding key: \" (ffirst kwbs)))))\n        (reduce process-entry [] bents)))))\n\n(core/defmacro goog-define\n  \"Defines a var using `goog.define`. Passed default value must be\n  string, number or boolean.\n\n  Default value can be overridden at compile time using the\n  compiler option `:closure-defines`.\n\n  Example:\n    (ns your-app.core)\n    (goog-define DEBUG! false)\n    ;; can be overridden with\n    :closure-defines {\\\"your_app.core.DEBUG_BANG_\\\" true}\n    or\n    :closure-defines {'your-app.core/DEBUG! true}\"\n  [sym default]\n  (assert-args goog-define\n   (core/or (core/string? default)\n            (core/number? default)\n            (core/true? default)\n            (core/false? default)) \"a string, number or boolean as default value\")\n  (core/let [defname (comp/munge (core/str *ns* \"/\" sym))\n             type    (core/cond\n                       (core/string? default) \"string\"\n                       (core/number? default) \"number\"\n                       (core/or (core/true? default) (core/false? default)) \"boolean\")]\n    `(do\n       (declare ~(symbol sym))\n       (~'js* ~(core/str \"/** @define {\" type \"} */\"))\n       (goog/define ~defname ~default))))\n\n(core/defmacro let\n  \"binding => binding-form init-expr\n\n  Evaluates the exprs in a lexical context in which the symbols in\n  the binding-forms are bound to their respective init-exprs or parts\n  therein.\"\n  [bindings & body]\n  (assert-args let\n     (vector? bindings) \"a vector for its binding\"\n     (even? (count bindings)) \"an even number of forms in binding vector\")\n  `(let* ~(destructure bindings) ~@body))\n\n(core/defmacro loop\n  \"Evaluates the exprs in a lexical context in which the symbols in\n  the binding-forms are bound to their respective init-exprs or parts\n  therein. Acts as a recur target.\"\n  [bindings & body]\n  (assert-args loop\n    (vector? bindings) \"a vector for its binding\"\n    (even? (count bindings)) \"an even number of forms in binding vector\")\n  (core/let [db (destructure bindings)]\n    (if (= db bindings)\n      `(loop* ~bindings ~@body)\n      (core/let [vs (take-nth 2 (drop 1 bindings))\n                 bs (take-nth 2 bindings)\n                 gs (map (core/fn [b] (if (core/symbol? b) b (gensym))) bs)\n                 bfs (reduce (core/fn [ret [b v g]]\n                               (if (core/symbol? b)\n                                 (conj ret g v)\n                                 (conj ret g v b g)))\n                       [] (map core/vector bs vs gs))]\n        `(let ~bfs\n           (loop* ~(vec (interleave gs gs))\n             (let ~(vec (interleave bs gs))\n               ~@body)))))))\n\n(def fast-path-protocols\n  \"protocol fqn -> [partition number, bit]\"\n  (zipmap (map #(symbol \"cljs.core\" (core/str %))\n               '[IFn ICounted IEmptyableCollection ICollection IIndexed ASeq ISeq INext\n                 ILookup IAssociative IMap IMapEntry ISet IStack IVector IDeref\n                 IDerefWithTimeout IMeta IWithMeta IReduce IKVReduce IEquiv IHash\n                 ISeqable ISequential IList IRecord IReversible ISorted IPrintWithWriter IWriter\n                 IPrintWithWriter IPending IWatchable IEditableCollection ITransientCollection\n                 ITransientAssociative ITransientMap ITransientVector ITransientSet\n                 IMultiFn IChunkedSeq IChunkedNext IComparable INamed ICloneable IAtom\n                 IReset ISwap IIterable])\n          (iterate (core/fn [[p b]]\n                     (if (core/== 2147483648 b)\n                       [(core/inc p) 1]\n                       [p #?(:clj  (core/bit-shift-left b 1)\n                             :cljs (core/* 2 b))]))\n                   [0 1])))\n\n(def fast-path-protocol-partitions-count\n  \"total number of partitions\"\n  (core/let [c (count fast-path-protocols)\n             m (core/mod c 32)]\n    (if (core/zero? m)\n      (core/quot c 32)\n      (core/inc (core/quot c 32)))))\n\n(core/defmacro str [& xs]\n  (core/let [strs (core/->> (repeat (count xs) \"cljs.core.str.cljs$core$IFn$_invoke$arity$1(~{})\")\n                    (interpose \",\")\n                    (apply core/str))]\n    (list* 'js* (core/str \"[\" strs \"].join('')\") xs)))\n\n(core/defn- bool-expr [e]\n  (vary-meta e assoc :tag 'boolean))\n\n(core/defn- simple-test-expr? [env ast]\n  (core/and\n    (#{:var :invoke :constant :dot :js} (:op ast))\n    ('#{boolean seq} (cljs.analyzer/infer-tag env ast))))\n\n(core/defmacro and\n  \"Evaluates exprs one at a time, from left to right. If a form\n  returns logical false (nil or false), and returns that value and\n  doesn't evaluate any of the other expressions, otherwise it returns\n  the value of the last expr. (and) returns true.\"\n  ([] true)\n  ([x] x)\n  ([x & next]\n   (core/let [forms (concat [x] next)]\n     (if (every? #(simple-test-expr? &env %)\n           (map #(cljs.analyzer/analyze &env %) forms))\n       (core/let [and-str (core/->> (repeat (count forms) \"(~{})\")\n                            (interpose \" && \")\n                            (apply core/str))]\n         (bool-expr `(~'js* ~and-str ~@forms)))\n       `(let [and# ~x]\n          (if and# (and ~@next) and#))))))\n\n(core/defmacro or\n  \"Evaluates exprs one at a time, from left to right. If a form\n  returns a logical true value, or returns that value and doesn't\n  evaluate any of the other expressions, otherwise it returns the\n  value of the last expression. (or) returns nil.\"\n  ([] nil)\n  ([x] x)\n  ([x & next]\n   (core/let [forms (concat [x] next)]\n     (if (every? #(simple-test-expr? &env %)\n           (map #(cljs.analyzer/analyze &env %) forms))\n       (core/let [or-str (core/->> (repeat (count forms) \"(~{})\")\n                           (interpose \" || \")\n                           (apply core/str))]\n         (bool-expr `(~'js* ~or-str ~@forms)))\n       `(let [or# ~x]\n          (if or# or# (or ~@next)))))))\n\n(core/defmacro nil? [x]\n  `(coercive-= ~x nil))\n\n(core/defmacro some? [x]\n  `(not (nil? ~x)))\n\n(core/defmacro coercive-not [x]\n  (bool-expr (core/list 'js* \"(!~{})\" x)))\n\n(core/defmacro coercive-not= [x y]\n  (bool-expr (core/list 'js* \"(~{} != ~{})\" x y)))\n\n(core/defmacro coercive-= [x y]\n  (bool-expr (core/list 'js* \"(~{} == ~{})\" x y)))\n\n(core/defmacro coercive-boolean [x]\n  (with-meta (core/list 'js* \"~{}\" x)\n    {:tag 'boolean}))\n\n;; internal - do not use.\n(core/defmacro truth_ [x]\n  (core/assert (core/symbol? x) \"x is substituted twice\")\n  (core/list 'js* \"(~{} != null && ~{} !== false)\" x x))\n\n(core/defmacro js-arguments []\n  (core/list 'js* \"arguments\"))\n\n(core/defmacro js-delete [obj key]\n  (core/list 'js* \"delete ~{}[~{}]\" obj key))\n\n(core/defmacro js-in [key obj]\n  (core/list 'js* \"~{} in ~{}\" key obj))\n\n(core/defmacro js-debugger\n  \"Emit JavaScript \\\"debugger;\\\" statement\"\n  []\n  (core/list 'do\n             (core/list 'js* \"debugger\")\n             nil))\n\n(core/defmacro js-comment\n  \"Emit a top-level JavaScript multi-line comment. New lines will create a\n  new comment line. Comment block will be preceded and followed by a newline\"\n  [comment]\n  (core/let [[x & ys] (string/split comment #\"\\n\")]\n    (core/list 'js*\n      (core/str\n        \"\\n/**\\n\"\n        (core/str \" * \" x \"\\n\")\n        (core/->> ys\n          (map #(core/str \" * \" (string/replace % #\"^   \" \"\") \"\\n\"))\n          (reduce core/str \"\"))\n        \" */\\n\"))))\n\n(core/defmacro unsafe-cast\n  \"EXPERIMENTAL: Subject to change. Unsafely cast a value to a different type.\"\n  [t x]\n  (core/let [cast-expr (core/str \"~{} = /** @type {\" t \"} */ (~{})\")]\n    (core/list 'js* cast-expr x x)))\n\n(core/defmacro js-inline-comment\n  \"Emit an inline JavaScript comment.\"\n  [comment]\n  (core/list 'js* (core/str \"/**\" comment \"*/\")))\n\n(core/defmacro true? [x]\n  (bool-expr (core/list 'js* \"~{} === true\" x)))\n\n(core/defmacro false? [x]\n  (bool-expr (core/list 'js* \"~{} === false\" x)))\n\n(core/defmacro string? [x]\n  (bool-expr (core/list 'js* \"typeof ~{} === 'string'\" x)))\n\n;; TODO: x must be a symbol, not an arbitrary expression\n(core/defmacro exists?\n  \"Return true if argument exists, analogous to usage of typeof operator\n   in JavaScript.\"\n  [x]\n  (bool-expr\n    (core/list 'js* \"typeof ~{} !== 'undefined'\"\n      (vary-meta x assoc :cljs.analyzer/no-resolve true))))\n\n(core/defmacro undefined?\n  \"Return true if argument is identical to the JavaScript undefined value.\"\n  [x]\n  (bool-expr (core/list 'js* \"(void 0 === ~{})\" x)))\n\n(core/defmacro identical? [a b]\n  (bool-expr (core/list 'js* \"(~{} === ~{})\" a b)))\n\n(core/defmacro instance? [c x]\n  ;; Google Closure warns about some references to RegExp, so\n  ;; (instance? RegExp ...) needs to be inlined, but the expansion\n  ;; should preserve the order of argument evaluation.\n  (bool-expr (if (clojure.core/symbol? c)\n               (core/list 'js* \"(~{} instanceof ~{})\" x c)\n               `(let [c# ~c x# ~x]\n                  (~'js* \"(~{} instanceof ~{})\" x# c#)))))\n\n(core/defmacro number? [x]\n  (bool-expr (core/list 'js* \"typeof ~{} === 'number'\" x)))\n\n(core/defmacro symbol? [x]\n  (bool-expr `(instance? Symbol ~x)))\n\n(core/defmacro keyword? [x]\n  (bool-expr `(instance? Keyword ~x)))\n\n(core/defmacro aget\n  ([a i]\n   (core/list 'js* \"(~{}[~{}])\" a i))\n  ([a i & idxs]\n   (core/let [astr (apply core/str (repeat (count idxs) \"[~{}]\"))]\n     `(~'js* ~(core/str \"(~{}[~{}]\" astr \")\") ~a ~i ~@idxs))))\n\n(core/defmacro aset\n  ([a i v]\n   (core/list 'js* \"(~{}[~{}] = ~{})\" a i v))\n  ([a idx idx2 & idxv]\n   (core/let [n    (core/dec (count idxv))\n              astr (apply core/str (repeat n \"[~{}]\"))]\n     `(~'js* ~(core/str \"(~{}[~{}][~{}]\" astr \" = ~{})\") ~a ~idx ~idx2 ~@idxv))))\n\n(core/defmacro ^::ana/numeric +\n  ([] 0)\n  ([x] x)\n  ([x y] (core/list 'js* \"(~{} + ~{})\" x y))\n  ([x y & more] `(+ (+ ~x ~y) ~@more)))\n\n(core/defmacro byte [x] x)\n(core/defmacro short [x] x)\n(core/defmacro float [x] x)\n(core/defmacro double [x] x)\n\n(core/defmacro unchecked-byte [x] x)\n(core/defmacro unchecked-char [x] x)\n(core/defmacro unchecked-short [x] x)\n(core/defmacro unchecked-float [x] x)\n(core/defmacro unchecked-double [x] x)\n\n(core/defmacro ^::ana/numeric unchecked-add\n  ([& xs] `(+ ~@xs)))\n\n(core/defmacro ^::ana/numeric unchecked-add-int\n  ([& xs] `(+ ~@xs)))\n\n(core/defmacro ^::ana/numeric unchecked-dec\n  ([x] `(dec ~x)))\n\n(core/defmacro ^::ana/numeric unchecked-dec-int\n  ([x] `(dec ~x)))\n\n(core/defmacro ^::ana/numeric unchecked-divide-int\n  ([& xs] `(/ ~@xs)))\n\n(core/defmacro ^::ana/numeric unchecked-inc\n  ([x] `(inc ~x)))\n\n(core/defmacro ^::ana/numeric unchecked-inc-int\n  ([x] `(inc ~x)))\n\n(core/defmacro ^::ana/numeric unchecked-multiply\n  ([& xs] `(* ~@xs)))\n\n(core/defmacro ^::ana/numeric unchecked-multiply-int\n  ([& xs] `(* ~@xs)))\n\n(core/defmacro ^::ana/numeric unchecked-negate\n  ([x] `(- ~x)))\n\n(core/defmacro ^::ana/numeric unchecked-negate-int\n  ([x] `(- ~x)))\n\n(core/defmacro ^::ana/numeric unchecked-remainder-int\n  ([x n] `(core/mod ~x ~n)))\n\n(core/defmacro ^::ana/numeric unchecked-subtract\n  ([& xs] `(- ~@xs)))\n\n(core/defmacro ^::ana/numeric unchecked-subtract-int\n  ([& xs] `(- ~@xs)))\n\n(core/defmacro ^::ana/numeric -\n  ([x] (core/list 'js* \"(- ~{})\" x))\n  ([x y] (core/list 'js* \"(~{} - ~{})\" x y))\n  ([x y & more] `(- (- ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric *\n  ([] 1)\n  ([x] x)\n  ([x y] (core/list 'js* \"(~{} * ~{})\" x y))\n  ([x y & more] `(* (* ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric /\n  ([x] `(/ 1 ~x))\n  ([x y] (core/list 'js* \"(~{} / ~{})\" x y))\n  ([x y & more] `(/ (/ ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric divide\n  ([x] `(/ 1 ~x))\n  ([x y] (core/list 'js* \"(~{} / ~{})\" x y))\n  ([x y & more] `(/ (/ ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric <\n  ([x] true)\n  ([x y] (bool-expr (core/list 'js* \"(~{} < ~{})\" x y)))\n  ([x y & more] `(and (< ~x ~y) (< ~y ~@more))))\n\n(core/defmacro ^::ana/numeric <=\n  ([x] true)\n  ([x y] (bool-expr (core/list 'js* \"(~{} <= ~{})\" x y)))\n  ([x y & more] `(and (<= ~x ~y) (<= ~y ~@more))))\n\n(core/defmacro ^::ana/numeric >\n  ([x] true)\n  ([x y] (bool-expr (core/list 'js* \"(~{} > ~{})\" x y)))\n  ([x y & more] `(and (> ~x ~y) (> ~y ~@more))))\n\n(core/defmacro ^::ana/numeric >=\n  ([x] true)\n  ([x y] (bool-expr (core/list 'js* \"(~{} >= ~{})\" x y)))\n  ([x y & more] `(and (>= ~x ~y) (>= ~y ~@more))))\n\n(core/defmacro ^::ana/numeric ==\n  ([x] true)\n  ([x y] (bool-expr (core/list 'js* \"(~{} === ~{})\" x y)))\n  ([x y & more] `(and (== ~x ~y) (== ~y ~@more))))\n\n(core/defmacro ^::ana/numeric dec [x]\n  `(- ~x 1))\n\n(core/defmacro ^::ana/numeric inc [x]\n  `(+ ~x 1))\n\n(core/defmacro ^::ana/numeric zero? [x]\n  `(== ~x 0))\n\n(core/defmacro ^::ana/numeric pos? [x]\n  `(> ~x 0))\n\n(core/defmacro ^::ana/numeric neg? [x]\n  `(< ~x 0))\n\n(core/defmacro ^::ana/numeric max\n  ([x] x)\n  ([x y] `(let [x# ~x, y# ~y]\n            (~'js* \"((~{} > ~{}) ? ~{} : ~{})\" x# y# x# y#)))\n  ([x y & more] `(max (max ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric min\n  ([x] x)\n  ([x y] `(let [x# ~x, y# ~y]\n            (~'js* \"((~{} < ~{}) ? ~{} : ~{})\" x# y# x# y#)))\n  ([x y & more] `(min (min ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric js-mod [num div]\n  (core/list 'js* \"(~{} % ~{})\" num div))\n\n(core/defmacro ^::ana/numeric bit-not [x]\n  (core/list 'js* \"(~ ~{})\" x))\n\n(core/defmacro ^::ana/numeric bit-and\n  ([x y] (core/list 'js* \"(~{} & ~{})\" x y))\n  ([x y & more] `(bit-and (bit-and ~x ~y) ~@more)))\n\n;; internal do not use\n(core/defmacro ^::ana/numeric unsafe-bit-and\n  ([x y] (bool-expr (core/list 'js* \"(~{} & ~{})\" x y)))\n  ([x y & more] `(unsafe-bit-and (unsafe-bit-and ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric bit-or\n  ([x y] (core/list 'js* \"(~{} | ~{})\" x y))\n  ([x y & more] `(bit-or (bit-or ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric int [x]\n  `(bit-or ~x 0))\n\n(core/defmacro ^::ana/numeric bit-xor\n  ([x y] (core/list 'js* \"(~{} ^ ~{})\" x y))\n  ([x y & more] `(bit-xor (bit-xor ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric bit-and-not\n  ([x y] (core/list 'js* \"(~{} & ~~{})\" x y))\n  ([x y & more] `(bit-and-not (bit-and-not ~x ~y) ~@more)))\n\n(core/defmacro ^::ana/numeric bit-clear [x n]\n  (core/list 'js* \"(~{} & ~(1 << ~{}))\" x n))\n\n(core/defmacro ^::ana/numeric bit-flip [x n]\n  (core/list 'js* \"(~{} ^ (1 << ~{}))\" x n))\n\n(core/defmacro bit-test [x n]\n  (bool-expr (core/list 'js* \"((~{} & (1 << ~{})) != 0)\" x n)))\n\n(core/defmacro ^::ana/numeric bit-shift-left [x n]\n  (core/list 'js* \"(~{} << ~{})\" x n))\n\n(core/defmacro ^::ana/numeric bit-shift-right [x n]\n  (core/list 'js* \"(~{} >> ~{})\" x n))\n\n(core/defmacro ^::ana/numeric bit-shift-right-zero-fill [x n]\n  (core/list 'js* \"(~{} >>> ~{})\" x n))\n\n(core/defmacro ^::ana/numeric unsigned-bit-shift-right [x n]\n  (core/list 'js* \"(~{} >>> ~{})\" x n))\n\n(core/defmacro ^::ana/numeric bit-set [x n]\n  (core/list 'js* \"(~{} | (1 << ~{}))\" x n))\n\n;; internal\n(core/defmacro mask [hash shift]\n  (core/list 'js* \"((~{} >>> ~{}) & 0x01f)\" hash shift))\n\n;; internal\n(core/defmacro bitpos [hash shift]\n  (core/list 'js* \"(1 << ~{})\" `(mask ~hash ~shift)))\n\n;; internal\n(core/defmacro caching-hash [coll hash-fn hash-key]\n  (core/assert (clojure.core/symbol? hash-key) \"hash-key is substituted twice\")\n  `(let [h# ~hash-key]\n     (if-not (nil? h#)\n       h#\n       (let [h# (~hash-fn ~coll)]\n         (set! ~hash-key h#)\n         h#))))\n\n;;; internal -- reducers-related macros\n\n(core/defn- do-curried\n  [name doc meta args body]\n  (core/let [cargs (vec (butlast args))]\n    `(defn ~name ~doc ~meta\n       (~cargs (fn [x#] (~name ~@cargs x#)))\n       (~args ~@body))))\n\n(core/defmacro ^:private defcurried\n  \"Builds another arity of the fn that returns a fn awaiting the last\n  param\"\n  [name doc meta args & body]\n  (do-curried name doc meta args body))\n\n(core/defn- do-rfn [f1 k fkv]\n  `(fn\n     ([] (~f1))\n     ~(clojure.walk/postwalk\n       #(if (sequential? %)\n          ((if (vector? %) vec identity)\n           (core/remove #{k} %))\n          %)\n       fkv)\n     ~fkv))\n\n(core/defmacro ^:private rfn\n  \"Builds 3-arity reducing fn given names of wrapped fn and key, and k/v impl.\"\n  [[f1 k] fkv]\n  (do-rfn f1 k fkv))\n\n;;; end of reducers macros\n\n(core/defn- protocol-prefix [psym]\n  (core/str (core/-> (core/str psym)\n              (.replace #?(:clj \\. :cljs (js/RegExp. \"\\\\.\" \"g\")) \\$)\n              (.replace \\/ \\$))\n    \"$\"))\n\n(def ^:private base-type\n     {nil \"null\"\n      'object \"object\"\n      'string \"string\"\n      'number \"number\"\n      'array \"array\"\n      'function \"function\"\n      'boolean \"boolean\"\n      'default \"_\"})\n\n(def ^:private js-base-type\n     {'js/Boolean \"boolean\"\n      'js/String \"string\"\n      'js/Array \"array\"\n      'js/Object \"object\"\n      'js/Number \"number\"\n      'js/Function \"function\"})\n\n(core/defmacro reify\n  \"reify is a macro with the following structure:\n\n (reify options* specs*)\n\n  Currently there are no options.\n\n  Each spec consists of the protocol name followed by zero\n  or more method bodies:\n\n  protocol\n  (methodName [args+] body)*\n\n  Methods should be supplied for all methods of the desired\n  protocol(s). You can also define overrides for Object methods. Note that\n  the first parameter must be supplied to correspond to the target object\n  ('this' in JavaScript parlance). Note also that recur calls\n  to the method head should *not* pass the target object, it will be supplied\n  automatically and can not be substituted.\n\n  recur works to method heads The method bodies of reify are lexical\n  closures, and can refer to the surrounding local scope:\n\n  (str (let [f \\\"foo\\\"]\n       (reify Object\n         (toString [this] f))))\n  == \\\"foo\\\"\n\n  (seq (let [f \\\"foo\\\"]\n       (reify ISeqable\n         (-seq [this] (seq f)))))\n  == (\\\"f\\\" \\\"o\\\" \\\"o\\\"))\n\n  reify always implements IMeta and IWithMeta and transfers meta\n  data of the form to the created object.\n\n  (meta ^{:k :v} (reify Object (toString [this] \\\"foo\\\")))\n  == {:k :v}\"\n  [& impls]\n  (core/let [t        (with-meta\n                        (gensym\n                          (core/str \"t_\"\n                            (string/replace (core/str (munge ana/*cljs-ns*)) \".\" \"$\")))\n                        {:anonymous true})\n             meta-sym (gensym \"meta\")\n             this-sym (gensym \"_\")\n             locals   (keys (:locals &env))\n             ns       (core/-> &env :ns :name)\n             munge    comp/munge]\n    `(do\n       (when-not (exists? ~(symbol (core/str ns) (core/str t)))\n         (deftype ~t [~@locals ~meta-sym]\n           IWithMeta\n           (~'-with-meta [~this-sym ~meta-sym]\n             (new ~t ~@locals ~meta-sym))\n           IMeta\n           (~'-meta [~this-sym] ~meta-sym)\n           ~@impls))\n       (new ~t ~@locals ~(ana/elide-reader-meta (meta &form))))))\n\n(core/defmacro specify!\n  \"Identical to reify but mutates its first argument.\"\n  [expr & impls]\n  (core/let [x (with-meta (gensym \"x\") {:extend :instance})]\n    `(let [~x ~expr]\n       (extend-type ~x ~@impls)\n       ~x)))\n\n(core/defmacro specify\n  \"Identical to specify! but does not mutate its first argument. The first\n  argument must be an ICloneable instance.\"\n  [expr & impls]\n  `(cljs.core/specify! (cljs.core/clone ~expr)\n     ~@impls))\n\n(core/defmacro ^:private js-this []\n  (core/list 'js* \"this\"))\n\n(core/defmacro this-as\n  \"Defines a scope where JavaScript's implicit \\\"this\\\" is bound to the name provided.\"\n  [name & body]\n  `(let [~name (js-this)]\n     ~@body))\n\n(core/defn- to-property [sym]\n  (symbol (core/str \"-\" sym)))\n\n(core/defn- warn-and-update-protocol [p type env]\n  (core/when-not (= 'Object p)\n    (core/if-let [var (cljs.analyzer/resolve-existing-var (dissoc env :locals) p)]\n      (do\n        (core/when-not (:protocol-symbol var)\n          (cljs.analyzer/warning :invalid-protocol-symbol env {:protocol p}))\n        (core/when (core/and (:protocol-deprecated cljs.analyzer/*cljs-warnings*)\n                (core/-> var :deprecated)\n                (not (core/-> p meta :deprecation-nowarn)))\n          (cljs.analyzer/warning :protocol-deprecated env {:protocol p}))\n        (core/when (:protocol-symbol var)\n          (swap! env/*compiler* update-in [:cljs.analyzer/namespaces]\n            (core/fn [ns]\n              (update-in ns [(:ns var) :defs (symbol (name p)) :impls]\n                conj type)))))\n      (core/when (:undeclared cljs.analyzer/*cljs-warnings*)\n        (cljs.analyzer/warning :undeclared-protocol-symbol env {:protocol p})))))\n\n(core/defn- resolve-var [env sym]\n  (core/let [ret (:name (cljs.analyzer/resolve-var env sym))]\n    (core/assert ret (core/str \"Can't resolve: \" sym))\n    ret))\n\n(core/defn- ->impl-map [impls]\n  (core/loop [ret {} s impls]\n    (if (seq s)\n      (recur (assoc ret (first s) (take-while seq? (next s)))\n        (drop-while seq? (next s)))\n      ret)))\n\n(core/defn- base-assign-impls [env resolve tsym type [p sigs]]\n  (warn-and-update-protocol p tsym env)\n  (core/let [psym       (resolve p)\n             pfn-prefix (subs (core/str psym) 0\n                          (clojure.core/inc (.indexOf (core/str psym) \"/\")))]\n    (cons `(aset ~psym ~type true)\n      (map (core/fn [[f & meths :as form]]\n             `(aset ~(symbol (core/str pfn-prefix f))\n                ~type ~(with-meta `(fn ~@meths) (meta form))))\n        sigs))))\n\n(core/defmulti ^:private extend-prefix (core/fn [tsym sym] (core/-> tsym meta :extend)))\n\n(core/defmethod extend-prefix :instance\n  [tsym sym] `(.. ~tsym ~(to-property sym)))\n\n(core/defmethod extend-prefix :default\n  [tsym sym] `(.. ~tsym ~'-prototype ~(to-property sym)))\n\n(core/defn- adapt-obj-params [type [[this & args :as sig] & body]]\n  (core/list (vec args)\n    (list* 'this-as (vary-meta this assoc :tag type) body)))\n\n(core/defn- adapt-ifn-params [type [[this & args :as sig] & body]]\n  (core/let [self-sym (with-meta 'self__ {:tag type})]\n    `(~(vec (cons self-sym args))\n       (this-as ~self-sym\n         (let [~this ~self-sym]\n           ~@body)))))\n\n;; for IFn invoke implementations, we need to drop first arg\n(core/defn- adapt-ifn-invoke-params [type [[this & args :as sig] & body]]\n  `(~(vec args)\n     (this-as ~(vary-meta this assoc :tag type)\n       ~@body)))\n\n(core/defn- adapt-proto-params [type [[this & args :as sig] & body]]\n  (core/let [this' (vary-meta this assoc :tag type)]\n    `(~(vec (cons this' args))\n      (this-as ~this'\n        ~@body))))\n\n(core/defn- add-obj-methods [type type-sym sigs]\n  (map (core/fn [[f & meths :as form]]\n         (core/let [[f meths] (if (vector? (first meths))\n                                [f [(rest form)]]\n                                [f meths])]\n           `(set! ~(extend-prefix type-sym f)\n              ~(with-meta `(fn ~@(map #(adapt-obj-params type %) meths)) (meta form)))))\n    sigs))\n\n(core/defn- ifn-invoke-methods [type type-sym [f & meths :as form]]\n  (map\n    (core/fn [meth]\n      (core/let [arity (count (first meth))]\n        `(set! ~(extend-prefix type-sym (symbol (core/str \"cljs$core$IFn$_invoke$arity$\" arity)))\n           ~(with-meta `(fn ~meth) (meta form)))))\n    (map #(adapt-ifn-invoke-params type %) meths)))\n\n(core/defn- add-ifn-methods [type type-sym [f & meths :as form]]\n  (core/let [meths    (map #(adapt-ifn-params type %) meths)\n             this-sym (with-meta 'self__ {:tag type})\n             argsym   (gensym \"args\")]\n    (concat\n      [`(set! ~(extend-prefix type-sym 'call) ~(with-meta `(fn ~@meths) (meta form)))\n       `(set! ~(extend-prefix type-sym 'apply)\n          ~(with-meta\n             `(fn ~[this-sym argsym]\n                (this-as ~this-sym\n                  (.apply (.-call ~this-sym) ~this-sym\n                    (.concat (array ~this-sym) (cljs.core/aclone ~argsym)))))\n             (meta form)))]\n      (ifn-invoke-methods type type-sym form))))\n\n(core/defn- add-proto-methods* [pprefix type type-sym [f & meths :as form]]\n  (core/let [pf (core/str pprefix (name f))]\n    (if (vector? (first meths))\n      ;; single method case\n      (core/let [meth meths]\n        [`(set! ~(extend-prefix type-sym (core/str pf \"$arity$\" (count (first meth))))\n            ~(with-meta `(fn ~@(adapt-proto-params type meth)) (meta form)))])\n      (map (core/fn [[sig & body :as meth]]\n             `(set! ~(extend-prefix type-sym (core/str pf \"$arity$\" (count sig)))\n                ~(with-meta `(fn ~(adapt-proto-params type meth)) (meta form))))\n        meths))))\n\n(core/defn- proto-assign-impls [env resolve type-sym type [p sigs]]\n  (warn-and-update-protocol p type env)\n  (core/let [psym      (resolve p)\n             pprefix   (protocol-prefix psym)\n             skip-flag (set (core/-> type-sym meta :skip-protocol-flag))]\n    (if (= p 'Object)\n      (add-obj-methods type type-sym sigs)\n      (concat\n        (core/when-not (skip-flag psym)\n          [`(set! ~(extend-prefix type-sym pprefix) cljs.core/PROTOCOL_SENTINEL)])\n        (mapcat\n          (core/fn [sig]\n            (if (= psym 'cljs.core/IFn)\n              (add-ifn-methods type type-sym sig)\n              (add-proto-methods* pprefix type type-sym sig)))\n          sigs)))))\n\n(core/defn- validate-impl-sigs [env p method]\n  (core/when-not (= p 'Object)\n    (core/let [var (ana/resolve-var (dissoc env :locals) p)\n               minfo (core/-> var :protocol-info :methods)\n               method-name (first method)\n               ->name (comp symbol name)\n               [fname sigs] (if (core/vector? (second method))\n                              [(->name method-name) [(second method)]]\n                              [(->name method-name) (map first (rest method))])\n               decmeths (core/get minfo fname ::not-found)]\n      (core/when (= decmeths ::not-found)\n        (ana/warning :protocol-invalid-method env {:protocol p :fname fname :no-such-method true}))\n      (core/when (namespace method-name)\n        (core/let [method-var (ana/resolve-var (dissoc env :locals) method-name\n                                ana/confirm-var-exist-warning)]\n          (core/when-not (= (:name var) (:protocol method-var))\n            (ana/warning :protocol-invalid-method env\n              {:protocol p :fname method-name :no-such-method true}))))\n      (core/loop [sigs sigs seen #{}]\n        (core/when (seq sigs)\n          (core/let [sig (first sigs)\n                     c   (count sig)]\n            (core/when (contains? seen c)\n              (ana/warning :protocol-duped-method env {:protocol p :fname fname}))\n            (core/when (some '#{&} sig)\n              (ana/warning :protocol-impl-with-variadic-method env {:protocol p :name fname}))\n            (core/when (core/and (not= decmeths ::not-found) (not (some #{c} (map count decmeths))))\n              (ana/warning :protocol-invalid-method env {:protocol p :fname fname :invalid-arity c}))\n            (recur (next sigs) (conj seen c))))))))\n\n(core/defn- validate-impls [env impls]\n  (core/loop [protos #{} impls impls]\n    (core/when (seq impls)\n      (core/let [proto   (first impls)\n                 methods (take-while seq? (next impls))\n                 impls   (drop-while seq? (next impls))]\n        (core/when (contains? protos proto)\n          (ana/warning :protocol-multiple-impls env {:protocol proto}))\n        (core/loop [seen #{} methods methods]\n          (core/when (seq methods)\n            (core/let [[fname :as method] (first methods)]\n              (core/when (contains? seen fname)\n                (ana/warning :extend-type-invalid-method-shape env\n                  {:protocol proto :method fname}))\n              (validate-impl-sigs env proto method)\n              (recur (conj seen fname) (next methods)))))\n        (recur (conj protos proto) impls)))))\n\n(core/defn- type-hint-first-arg\n  [type-sym argv]\n  (assoc argv 0 (vary-meta (argv 0) assoc :tag type-sym)))\n\n(core/defn- type-hint-single-arity-sig\n  [type-sym sig]\n  (list* (first sig) (type-hint-first-arg type-sym (second sig)) (nnext sig)))\n\n(core/defn- type-hint-multi-arity-sig\n  [type-sym sig]\n  (list* (type-hint-first-arg type-sym (first sig)) (next sig)))\n\n(core/defn- type-hint-multi-arity-sigs\n  [type-sym sigs]\n  (list* (first sigs) (map (partial type-hint-multi-arity-sig type-sym) (rest sigs))))\n\n(core/defn- type-hint-sigs\n  [type-sym sig]\n  (if (vector? (second sig))\n    (type-hint-single-arity-sig type-sym sig)\n    (type-hint-multi-arity-sigs type-sym sig)))\n\n(core/defn- type-hint-impl-map\n  [type-sym impl-map]\n  (reduce-kv (core/fn [m proto sigs]\n               (assoc m proto (map (partial type-hint-sigs type-sym) sigs)))\n    {} impl-map))\n\n(core/defmacro extend-type\n  \"Extend a type to a series of protocols. Useful when you are\n  supplying the definitions explicitly inline. Propagates the\n  type as a type hint on the first argument of all fns.\n\n  type-sym may be\n\n   * default, meaning the definitions will apply for any value,\n     unless an extend-type exists for one of the more specific\n     cases below.\n   * nil, meaning the definitions will apply for the nil value.\n   * any of object, boolean, number, string, array, or function,\n     indicating the definitions will apply for values of the\n     associated base JavaScript types. Note that, for example,\n     string should be used instead of js/String.\n   * a JavaScript type not covered by the previous list, such\n     as js/RegExp.\n   * a type defined by deftype or defrecord.\n\n  (extend-type MyType\n    ICounted\n    (-count [c] ...)\n    Foo\n    (bar [x y] ...)\n    (baz ([x] ...) ([x y] ...) ...)\"\n  [type-sym & impls]\n  (core/let [env &env\n             _ (validate-impls env impls)\n             resolve (partial resolve-var env)\n             impl-map (->impl-map impls)\n             impl-map (if ('#{boolean number} type-sym)\n                        (type-hint-impl-map type-sym impl-map)\n                        impl-map)\n             [type assign-impls] (core/if-let [type (base-type type-sym)]\n                                   [type base-assign-impls]\n                                   [(resolve type-sym) proto-assign-impls])]\n    (core/when (core/and (:extending-base-js-type cljs.analyzer/*cljs-warnings*)\n            (js-base-type type-sym))\n      (cljs.analyzer/warning :extending-base-js-type env\n        {:current-symbol type-sym :suggested-symbol (js-base-type type-sym)}))\n    `(do ~@(mapcat #(assign-impls env resolve type-sym type %) impl-map))))\n\n(core/defn- prepare-protocol-masks [env impls]\n  (core/let [resolve  (partial resolve-var env)\n             impl-map (->impl-map impls)\n             fpp-pbs  (seq\n                        (keep fast-path-protocols\n                          (map resolve\n                            (keys impl-map))))]\n    (if fpp-pbs\n      (core/let [fpps  (into #{}\n                         (filter (partial contains? fast-path-protocols)\n                           (map resolve (keys impl-map))))\n                 parts (core/as-> (group-by first fpp-pbs) parts\n                         (into {}\n                           (map (juxt key (comp (partial map peek) val))\n                             parts))\n                         (into {}\n                           (map (juxt key (comp (partial reduce core/bit-or) val))\n                             parts)))]\n        [fpps (reduce (core/fn [ps p] (update-in ps [p] (core/fnil identity 0)))\n                parts\n                (range fast-path-protocol-partitions-count))]))))\n\n(core/defn- annotate-specs [annots v [f sigs]]\n  (conj v\n    (vary-meta (cons f (map #(cons (second %) (nnext %)) sigs))\n      merge annots)))\n\n(core/defn dt->et\n  ([type specs fields]\n   (dt->et type specs fields false))\n  ([type specs fields inline]\n   (core/let [annots {:cljs.analyzer/type type\n                      :cljs.analyzer/protocol-impl true\n                      :cljs.analyzer/protocol-inline inline}]\n     (core/loop [ret [] specs specs]\n       (if (seq specs)\n         (core/let [p     (first specs)\n                    ret   (core/-> (conj ret p)\n                            (into (reduce (partial annotate-specs annots) []\n                                    (group-by first (take-while seq? (next specs))))))\n                    specs (drop-while seq? (next specs))]\n           (recur ret specs))\n         ret)))))\n\n(core/defn- collect-protocols [impls env]\n  (core/->> impls\n      (filter core/symbol?)\n      (map #(:name (cljs.analyzer/resolve-var (dissoc env :locals) %)))\n      (into #{})))\n\n(core/defn- build-positional-factory\n  [rsym rname fields]\n  (core/let [fn-name (with-meta (symbol (core/str '-> rsym))\n                       (assoc (meta rsym) :factory :positional))\n        field-values (if (core/-> rsym meta :internal-ctor) (conj fields nil nil nil) fields)]\n    `(defn ~fn-name\n       [~@fields]\n       (new ~rname ~@field-values))))\n\n(core/defn- validate-fields\n  [case name fields]\n  (core/when-not (vector? fields)\n    (throw\n      #?(:clj (AssertionError. (core/str case \" \" name \", no fields vector given.\"))\n         :cljs (js/Error. (core/str case \" \" name \", no fields vector given.\"))))))\n\n(core/defmacro deftype\n  \"(deftype name [fields*]  options* specs*)\n\n  Currently there are no options.\n\n  Each spec consists of a protocol or interface name followed by zero\n  or more method bodies:\n\n  protocol-or-Object\n  (methodName [args*] body)*\n\n  The type will have the (by default, immutable) fields named by\n  fields, which can have type hints. Protocols and methods\n  are optional. The only methods that can be supplied are those\n  declared in the protocols/interfaces.  Note that method bodies are\n  not closures, the local environment includes only the named fields,\n  and those fields can be accessed directly. Fields can be qualified\n  with the metadata :mutable true at which point (set! afield aval) will be\n  supported in method bodies. Note well that mutable fields are extremely\n  difficult to use correctly, and are present only to facilitate the building\n  of higherlevel constructs, such as ClojureScript's reference types, in\n  ClojureScript itself. They are for experts only - if the semantics and\n  implications of :mutable are not immediately apparent to you, you should not\n  be using them.\n\n  Method definitions take the form:\n\n  (methodname [args*] body)\n\n  The argument and return types can be hinted on the arg and\n  methodname symbols. If not supplied, they will be inferred, so type\n  hints should be reserved for disambiguation.\n\n  Methods should be supplied for all methods of the desired\n  protocol(s). You can also define overrides for methods of Object. Note that\n  a parameter must be supplied to correspond to the target object\n  ('this' in JavaScript parlance). Note also that recur calls to the method\n  head should *not* pass the target object, it will be supplied\n  automatically and can not be substituted.\n\n  In the method bodies, the (unqualified) name can be used to name the\n  class (for calls to new, instance? etc).\n\n  One constructor will be defined, taking the designated fields.  Note\n  that the field names __meta and __extmap are currently reserved and\n  should not be used when defining your own types.\n\n  Given (deftype TypeName ...), a factory function called ->TypeName\n  will be defined, taking positional parameters for the fields\"\n  [t fields & impls]\n  (validate-fields \"deftype\" t fields)\n  (core/let [env &env\n             r (:name (cljs.analyzer/resolve-var (dissoc env :locals) t))\n             [fpps pmasks] (prepare-protocol-masks env impls)\n             protocols (collect-protocols impls env)\n             t (vary-meta t assoc\n                 :protocols protocols\n                 :skip-protocol-flag fpps) ]\n    `(do\n       (deftype* ~t ~fields ~pmasks\n         ~(if (seq impls)\n            `(extend-type ~t ~@(dt->et t impls fields))))\n       (set! (.-getBasis ~t) (fn [] '[~@fields]))\n       (set! (.-cljs$lang$type ~t) true)\n       (set! (.-cljs$lang$ctorStr ~t) ~(core/str r))\n       (set! (.-cljs$lang$ctorPrWriter ~t) (fn [this# writer# opt#] (-write writer# ~(core/str r))))\n\n       ~(build-positional-factory t r fields)\n       ~t)))\n\n(core/defn- emit-defrecord\n  \"Do not use this directly - use defrecord\"\n  [env tagname rname fields impls]\n  (core/let [hinted-fields fields\n             fields (vec (map #(with-meta % nil) fields))\n             base-fields fields\n             pr-open (core/str \"#\" #?(:clj  (.getNamespace rname)\n                                      :cljs (namespace rname))\n                               \".\" #?(:clj  (.getName rname)\n                                      :cljs (name rname))\n                               \"{\")\n             fields (conj fields '__meta '__extmap (with-meta '__hash {:mutable true}))]\n    (core/let [gs (gensym)\n               ksym (gensym \"k\")\n               impls (concat\n                       impls\n                       ['IRecord\n                        'ICloneable\n                        `(~'-clone [this#] (new ~tagname ~@fields))\n                        'IHash\n                        `(~'-hash [this#]\n                           (caching-hash this#\n                             (fn [coll#]\n                               (bit-xor\n                                 ~(hash (core/-> rname comp/munge core/str))\n                                 (hash-unordered-coll coll#)))\n                             ~'__hash))\n                        'IEquiv\n                        (core/let [this (gensym 'this) other (gensym 'other)]\n                          `(~'-equiv [~this ~other]\n                             (and (some? ~other)\n                                  (identical? (.-constructor ~this)\n                                              (.-constructor ~other))\n                                  ~@(map (core/fn [field]\n                                           `(= (.. ~this ~(to-property field))\n                                               (.. ~other ~(to-property field))))\n                                         base-fields)\n                                  (= (.-__extmap ~this)\n                                     (.-__extmap ~other)))))\n                        'IMeta\n                        `(~'-meta [this#] ~'__meta)\n                        'IWithMeta\n                        `(~'-with-meta [this# ~gs] (new ~tagname ~@(replace {'__meta gs} fields)))\n                        'ILookup\n                        `(~'-lookup [this# k#] (-lookup this# k# nil))\n                        `(~'-lookup [this# ~ksym else#]\n                           (case ~ksym\n                             ~@(mapcat (core/fn [f] [(keyword f) f]) base-fields)\n                             (cljs.core/get ~'__extmap ~ksym else#)))\n                        'ICounted\n                        `(~'-count [this#] (+ ~(count base-fields) (count ~'__extmap)))\n                        'ICollection\n                        `(~'-conj [this# entry#]\n                           (if (vector? entry#)\n                             (-assoc this# (-nth entry# 0) (-nth entry# 1))\n                             (reduce -conj\n                               this#\n                               entry#)))\n                        'IAssociative\n                        `(~'-assoc [this# k# ~gs]\n                           (condp keyword-identical? k#\n                             ~@(mapcat (core/fn [fld]\n                                         [(keyword fld) (list* `new tagname (replace {fld gs '__hash nil} fields))])\n                                 base-fields)\n                             (new ~tagname ~@(remove #{'__extmap '__hash} fields) (assoc ~'__extmap k# ~gs) nil)))\n                        'IMap\n                        `(~'-dissoc [this# k#] (if (contains? #{~@(map keyword base-fields)} k#)\n                                                 (dissoc (-with-meta (into {} this#) ~'__meta) k#)\n                                                 (new ~tagname ~@(remove #{'__extmap '__hash} fields)\n                                                   (not-empty (dissoc ~'__extmap k#))\n                                                   nil)))\n                        'ISeqable\n                        `(~'-seq [this#] (seq (concat [~@(map #(core/list `vector (keyword %) %) base-fields)]\n                                                ~'__extmap)))\n\n                        'IIterable\n                        `(~'-iterator [~gs]\n                          (RecordIter. 0 ~gs ~(count base-fields) [~@(map keyword base-fields)] (if ~'__extmap\n                                                                                                  (-iterator ~'__extmap)\n                                                                                                  (core/nil-iter))))\n\n                        'IPrintWithWriter\n                        `(~'-pr-writer [this# writer# opts#]\n                           (let [pr-pair# (fn [keyval#] (pr-sequential-writer writer# pr-writer \"\" \" \" \"\" opts# keyval#))]\n                             (pr-sequential-writer\n                               writer# pr-pair# ~pr-open \", \" \"}\" opts#\n                               (concat [~@(map #(core/list `vector (keyword %) %) base-fields)]\n                                 ~'__extmap))))\n                        ])\n               [fpps pmasks] (prepare-protocol-masks env impls)\n               protocols (collect-protocols impls env)\n               tagname (vary-meta tagname assoc\n                         :protocols protocols\n                         :skip-protocol-flag fpps)]\n      `(do\n         (~'defrecord* ~tagname ~hinted-fields ~pmasks\n           (extend-type ~tagname ~@(dt->et tagname impls fields true)))))))\n\n(core/defn- build-map-factory [rsym rname fields]\n  (core/let [fn-name (with-meta (symbol (core/str 'map-> rsym))\n                       (assoc (meta rsym) :factory :map))\n             ms (gensym)\n             ks (map keyword fields)\n             getters (map (core/fn [k] `(~k ~ms)) ks)]\n    `(defn ~fn-name [~ms]\n       (new ~rname ~@getters nil (not-empty (dissoc ~ms ~@ks)) nil))))\n\n(core/defmacro defrecord\n  \"(defrecord name [fields*]  options* specs*)\n\n  Currently there are no options.\n\n  Each spec consists of a protocol or interface name followed by zero\n  or more method bodies:\n\n  protocol-or-Object\n  (methodName [args*] body)*\n\n  The record will have the (immutable) fields named by\n  fields, which can have type hints. Protocols and methods\n  are optional. The only methods that can be supplied are those\n  declared in the protocols.  Note that method bodies are\n  not closures, the local environment includes only the named fields,\n  and those fields can be accessed directly.\n\n  Method definitions take the form:\n\n  (methodname [args*] body)\n\n  The argument and return types can be hinted on the arg and\n  methodname symbols. If not supplied, they will be inferred, so type\n  hints should be reserved for disambiguation.\n\n  Methods should be supplied for all methods of the desired\n  protocol(s). You can also define overrides for\n  methods of Object. Note that a parameter must be supplied to\n  correspond to the target object ('this' in JavaScript parlance). Note also\n  that recur calls to the method head should *not* pass the target object, it\n  will be supplied automatically and can not be substituted.\n\n  In the method bodies, the (unqualified) name can be used to name the\n  class (for calls to new, instance? etc).\n\n  The type will have implementations of several ClojureScript\n  protocol generated automatically: IMeta/IWithMeta (metadata support) and\n  IMap, etc.\n\n  In addition, defrecord will define type-and-value-based =,\n  and will define ClojureScript IHash and IEquiv.\n\n  Two constructors will be defined, one taking the designated fields\n  followed by a metadata map (nil for none) and an extension field\n  map (nil for none), and one taking only the fields (using nil for\n  meta and extension fields). Note that the field names __meta\n  and __extmap are currently reserved and should not be used when\n  defining your own records.\n\n  Given (defrecord TypeName ...), two factory functions will be\n  defined: ->TypeName, taking positional parameters for the fields,\n  and map->TypeName, taking a map of keywords to field values.\"\n  [rsym fields & impls]\n  (validate-fields \"defrecord\" rsym fields)\n  (core/let [rsym (vary-meta rsym assoc :internal-ctor true)\n             r    (vary-meta\n                    (:name (cljs.analyzer/resolve-var (dissoc &env :locals) rsym))\n                    assoc :internal-ctor true)]\n    `(let []\n       ~(emit-defrecord &env rsym r fields impls)\n       (set! (.-getBasis ~r) (fn [] '[~@fields]))\n       (set! (.-cljs$lang$type ~r) true)\n       (set! (.-cljs$lang$ctorPrSeq ~r) (fn [this#] (cljs.core/list ~(core/str r))))\n       (set! (.-cljs$lang$ctorPrWriter ~r) (fn [this# writer#] (-write writer# ~(core/str r))))\n       ~(build-positional-factory rsym r fields)\n       ~(build-map-factory rsym r fields)\n       ~r)))\n\n(core/defmacro defprotocol\n  \"A protocol is a named set of named methods and their signatures:\n\n  (defprotocol AProtocolName\n    ;optional doc string\n    \\\"A doc string for AProtocol abstraction\\\"\n\n  ;method signatures\n    (bar [this a b] \\\"bar docs\\\")\n    (baz [this a] [this a b] [this a b c] \\\"baz docs\\\"))\n\n  No implementations are provided. Docs can be specified for the\n  protocol overall and for each method. The above yields a set of\n  polymorphic functions and a protocol object. All are\n  namespace-qualified by the ns enclosing the definition The resulting\n  functions dispatch on the type of their first argument, which is\n  required and corresponds to the implicit target object ('this' in\n  JavaScript parlance). defprotocol is dynamic, has no special compile-time\n  effect, and defines no new types.\n\n  (defprotocol P\n    (foo [this])\n    (bar-me [this] [this y]))\n\n  (deftype Foo [a b c]\n    P\n    (foo [this] a)\n    (bar-me [this] b)\n    (bar-me [this y] (+ c y)))\n\n  (bar-me (Foo. 1 2 3) 42)\n  => 45\n\n  (foo\n    (let [x 42]\n      (reify P\n        (foo [this] 17)\n        (bar-me [this] x)\n        (bar-me [this y] x))))\n  => 17\"\n  [psym & doc+methods]\n  (core/let [p (:name (cljs.analyzer/resolve-var (dissoc &env :locals) psym))\n             [doc methods] (if (core/string? (first doc+methods))\n                             [(first doc+methods) (next doc+methods)]\n                             [nil doc+methods])\n             psym (vary-meta psym assoc\n                    :doc doc\n                    :protocol-symbol true)\n             ns-name (core/-> &env :ns :name)\n             fqn (core/fn [n] (symbol (core/str ns-name \".\" n)))\n             prefix (protocol-prefix p)\n             _ (core/doseq [[mname & arities] methods]\n                 (core/when (some #{0} (map count (filter vector? arities)))\n                   (throw\n                     #?(:clj (Exception.\n                               (core/str \"Invalid protocol, \" psym\n                                 \" defines method \" mname \" with arity 0\"))\n                        :cljs (js/Error.\n                                (core/str \"Invalid protocol, \" psym\n                                  \" defines method \" mname \" with arity 0\"))))))\n             expand-sig (core/fn [fname slot sig]\n                          (core/let [sig (core/if-not (every? core/symbol? sig)\n                                           (mapv (core/fn [arg]\n                                                   (core/cond\n                                                     (core/symbol? arg) arg\n                                                     (core/and (map? arg) (core/some? (:as arg))) (:as arg)\n                                                     :else (gensym))) sig)\n                                           sig)]\n                            `(~sig\n                              (if (and (not (nil? ~(first sig)))\n                                    (not (nil? (. ~(first sig) ~(symbol (core/str \"-\" slot)))))) ;; Property access needed here.\n                                (. ~(first sig) ~slot ~@sig)\n                                (let [x# (if (nil? ~(first sig)) nil ~(first sig))\n                                      m# (aget ~(fqn fname) (goog/typeOf x#))]\n                                  (if-not (nil? m#)\n                                    (m# ~@sig)\n                                    (let [m# (aget ~(fqn fname) \"_\")]\n                                      (if-not (nil? m#)\n                                        (m# ~@sig)\n                                        (throw\n                                          (missing-protocol\n                                            ~(core/str psym \".\" fname) ~(first sig)))))))))))\n             psym (core/-> psym\n                    (vary-meta update-in [:jsdoc] conj\n                      \"@interface\")\n                    (vary-meta assoc-in [:protocol-info :methods]\n                      (into {}\n                        (map\n                          (core/fn [[fname & sigs]]\n                            (core/let [doc (core/as-> (last sigs) doc\n                                             (core/when (core/string? doc) doc))\n                                       sigs (take-while vector? sigs)]\n                              [(vary-meta fname assoc :doc doc)\n                               (vec sigs)]))\n                          methods))))\n             method (core/fn [[fname & sigs]]\n                      (core/let [doc (core/as-> (last sigs) doc\n                                       (core/when (core/string? doc) doc))\n                                 sigs (take-while vector? sigs)\n                                 amp (core/when (some #{'&} (apply concat sigs))\n                                       (cljs.analyzer/warning\n                                        :protocol-with-variadic-method\n                                        &env {:protocol psym :name fname}))\n                                 slot (symbol (core/str prefix (name fname)))\n                                 fname (vary-meta fname assoc\n                                         :protocol p\n                                         :doc doc)]\n                        `(defn ~fname\n                           ~@(map (core/fn [sig]\n                                    (expand-sig fname\n                                      (symbol (core/str slot \"$arity$\" (count sig)))\n                                      sig))\n                               sigs))))]\n    `(do\n       (set! ~'*unchecked-if* true)\n       (def ~psym (~'js* \"function(){}\"))\n       ~@(map method methods)\n       (set! ~'*unchecked-if* false))))\n\n(core/defmacro implements?\n  \"EXPERIMENTAL\"\n  [psym x]\n  (core/let [p          (:name\n                          (cljs.analyzer/resolve-var\n                            (dissoc &env :locals) psym))\n             prefix     (protocol-prefix p)\n             xsym       (bool-expr (gensym))\n             [part bit] (fast-path-protocols p)\n             msym       (symbol\n                          (core/str \"-cljs$lang$protocol_mask$partition\" part \"$\"))]\n    (core/if-not (core/symbol? x)\n      `(let [~xsym ~x]\n         (if ~xsym\n           (if (or ~(if bit `(unsafe-bit-and (. ~xsym ~msym) ~bit) false)\n                    (identical? cljs.core/PROTOCOL_SENTINEL (. ~xsym ~(symbol (core/str \"-\" prefix)))))\n             true\n             false)\n           false))\n      `(if-not (nil? ~x)\n         (if (or ~(if bit `(unsafe-bit-and (. ~x ~msym) ~bit) false)\n                  (identical? cljs.core/PROTOCOL_SENTINEL (. ~x ~(symbol (core/str \"-\" prefix)))))\n           true\n           false)\n         false))))\n\n(core/defmacro satisfies?\n  \"Returns true if x satisfies the protocol\"\n  [psym x]\n  (core/let [p          (:name\n                          (cljs.analyzer/resolve-var\n                            (dissoc &env :locals) psym))\n             prefix     (protocol-prefix p)\n             xsym       (bool-expr (gensym))\n             [part bit] (fast-path-protocols p)\n             msym       (symbol\n                          (core/str \"-cljs$lang$protocol_mask$partition\" part \"$\"))]\n    (core/if-not (core/symbol? x)\n      `(let [~xsym ~x]\n         (if-not (nil? ~xsym)\n           (if (or ~(if bit `(unsafe-bit-and (. ~xsym ~msym) ~bit) false)\n                    (identical? cljs.core/PROTOCOL_SENTINEL (. ~xsym ~(symbol (core/str \"-\" prefix)))))\n             true\n             (if (coercive-not (. ~xsym ~msym))\n               (cljs.core/native-satisfies? ~psym ~xsym)\n               false))\n           (cljs.core/native-satisfies? ~psym ~xsym)))\n      `(if-not (nil? ~x)\n         (if (or ~(if bit `(unsafe-bit-and (. ~x ~msym) ~bit) false)\n                  (identical? cljs.core/PROTOCOL_SENTINEL (. ~x ~(symbol (core/str \"-\" prefix)))))\n           true\n           (if (coercive-not (. ~x ~msym))\n             (cljs.core/native-satisfies? ~psym ~x)\n             false))\n         (cljs.core/native-satisfies? ~psym ~x)))))\n\n(core/defmacro lazy-seq\n  \"Takes a body of expressions that returns an ISeq or nil, and yields\n  a ISeqable object that will invoke the body only the first time seq\n  is called, and will cache the result and return it on all subsequent\n  seq calls.\"\n  [& body]\n  `(new cljs.core/LazySeq nil (fn [] ~@body) nil nil))\n\n(core/defmacro delay\n  \"Takes a body of expressions and yields a Delay object that will\n  invoke the body only the first time it is forced (with force or deref/@), and\n  will cache the result and return it on all subsequent force\n  calls.\"\n  [& body]\n  `(new cljs.core/Delay (fn [] ~@body) nil))\n\n(core/defmacro with-redefs\n  \"binding => var-symbol temp-value-expr\n\n  Temporarily redefines vars while executing the body.  The\n  temp-value-exprs will be evaluated and each resulting value will\n  replace in parallel the root value of its var.  After the body is\n  executed, the root values of all the vars will be set back to their\n  old values. Useful for mocking out functions during testing.\"\n  [bindings & body]\n  (core/let [names (take-nth 2 bindings)\n             vals (take-nth 2 (drop 1 bindings))\n             tempnames (map (comp gensym name) names)\n             binds (map core/vector names vals)\n             resets (reverse (map core/vector names tempnames))\n             bind-value (core/fn [[k v]] (core/list 'set! k v))]\n    `(let [~@(interleave tempnames names)]\n       ~@(map bind-value binds)\n       (try\n         ~@body\n         (finally\n           ~@(map bind-value resets))))))\n\n(core/defmacro binding\n  \"binding => var-symbol init-expr\n\n  Creates new bindings for the (already-existing) vars, with the\n  supplied initial values, executes the exprs in an implicit do, then\n  re-establishes the bindings that existed before.  The new bindings\n  are made in parallel (unlike let); all init-exprs are evaluated\n  before the vars are bound to their new values.\"\n  [bindings & body]\n  (core/let [names (take-nth 2 bindings)]\n    (cljs.analyzer/confirm-bindings &env names)\n    `(with-redefs ~bindings ~@body)))\n\n(core/defmacro condp\n  \"Takes a binary predicate, an expression, and a set of clauses.\n  Each clause can take the form of either:\n\n  test-expr result-expr\n\n  test-expr :>> result-fn\n\n  Note :>> is an ordinary keyword.\n\n  For each clause, (pred test-expr expr) is evaluated. If it returns\n  logical true, the clause is a match. If a binary clause matches, the\n  result-expr is returned, if a ternary clause matches, its result-fn,\n  which must be a unary function, is called with the result of the\n  predicate as its argument, the result of that call being the return\n  value of condp. A single default expression can follow the clauses,\n  and its value will be returned if no clause matches. If no default\n  expression is provided and no clause matches, an\n  IllegalArgumentException is thrown.\"\n  {:added \"1.0\"}\n\n  [pred expr & clauses]\n  (core/let [gpred (gensym \"pred__\")\n             gexpr (gensym \"expr__\")\n             emit (core/fn emit [pred expr args]\n                    (core/let [[[a b c :as clause] more]\n                               (split-at (if (= :>> (second args)) 3 2) args)\n                               n (count clause)]\n                      (core/cond\n                        (= 0 n) `(throw (js/Error. (cljs.core/str \"No matching clause: \" ~expr)))\n                        (= 1 n) a\n                        (= 2 n) `(if (~pred ~a ~expr)\n                                   ~b\n                                   ~(emit pred expr more))\n                        :else `(if-let [p# (~pred ~a ~expr)]\n                                 (~c p#)\n                                 ~(emit pred expr more)))))\n             gres (gensym \"res__\")]\n    `(let [~gpred ~pred\n           ~gexpr ~expr]\n       ~(emit gpred gexpr clauses))))\n\n(core/defn- assoc-test [m test expr env]\n  (if (contains? m test)\n    (throw\n      #?(:clj (clojure.core/IllegalArgumentException.\n                (core/str \"Duplicate case test constant '\"\n                  test \"'\"\n                  (core/when (:line env)\n                    (core/str \" on line \" (:line env) \" \"\n                      cljs.analyzer/*cljs-file*))))\n         :cljs (js/Error.\n                 (core/str \"Duplicate case test constant '\"\n                   test \"'\"\n                   (core/when (:line env)\n                     (core/str \" on line \" (:line env) \" \"\n                       cljs.analyzer/*cljs-file*))))))\n    (assoc m test expr)))\n\n(core/defn- const? [env x]\n  (core/let [m (core/and (core/list? x)\n                         (ana/resolve-var env (last x)))]\n    (core/when m (core/get m :const))))\n\n(core/defmacro case\n  \"Takes an expression, and a set of clauses.\n\n  Each clause can take the form of either:\n\n  test-constant result-expr\n\n  (test-constant1 ... test-constantN)  result-expr\n\n  The test-constants are not evaluated. They must be compile-time\n  literals, and need not be quoted.  If the expression is equal to a\n  test-constant, the corresponding result-expr is returned. A single\n  default expression can follow the clauses, and its value will be\n  returned if no clause matches. If no default expression is provided\n  and no clause matches, an Error is thrown.\n\n  Unlike cond and condp, case does a constant-time dispatch, the\n  clauses are not considered sequentially.  All manner of constant\n  expressions are acceptable in case, including numbers, strings,\n  symbols, keywords, and (ClojureScript) composites thereof. Note that since\n  lists are used to group multiple constants that map to the same\n  expression, a vector can be used to match a list if needed. The\n  test-constants need not be all of the same type.\"\n  [e & clauses]\n  (core/let [esym    (gensym)\n             default (if (odd? (count clauses))\n                       (last clauses)\n                       `(throw\n                          (js/Error.\n                            (cljs.core/str \"No matching clause: \" ~esym))))\n             env     &env\n             pairs   (reduce\n                       (core/fn [m [test expr]]\n                         (core/cond\n                           (seq? test)\n                           (reduce\n                             (core/fn [m test]\n                               (core/let [test (if (core/symbol? test)\n                                                 (core/list 'quote test)\n                                                 test)]\n                                 (assoc-test m test expr env)))\n                             m test)\n                           (core/symbol? test)\n                           (assoc-test m (core/list 'quote test) expr env)\n                           :else\n                           (assoc-test m test expr env)))\n                     {} (partition 2 clauses))\n             tests   (keys pairs)]\n    (core/cond\n      (every? (some-fn core/number? core/string? #?(:clj core/char? :cljs (core/fnil core/char? :nonchar)) #(const? env %)) tests)\n      (core/let [no-default (if (odd? (count clauses)) (butlast clauses) clauses)\n                 tests      (mapv #(if (seq? %) (vec %) [%]) (take-nth 2 no-default))\n                 thens      (vec (take-nth 2 (drop 1 no-default)))]\n        `(let [~esym ~e] (case* ~esym ~tests ~thens ~default)))\n\n      (every? core/keyword? tests)\n      (core/let [no-default (if (odd? (count clauses)) (butlast clauses) clauses)\n                 kw-str #(.substring (core/str %) 1)\n                 tests (mapv #(if (seq? %) (mapv kw-str %) [(kw-str %)]) (take-nth 2 no-default))\n                 thens (vec (take-nth 2 (drop 1 no-default)))]\n        `(let [~esym ~e\n               ~esym (if (keyword? ~esym) (.-fqn ~esym) nil)]\n           (case* ~esym ~tests ~thens ~default)))\n\n      ;; equality\n      :else\n      `(let [~esym ~e]\n         (cond\n           ~@(mapcat (core/fn [[m c]] `((cljs.core/= ~m ~esym) ~c)) pairs)\n           :else ~default)))))\n\n(core/defmacro assert\n  \"Evaluates expr and throws an exception if it does not evaluate to\n  logical true.\"\n  ([x]\n     (core/when *assert*\n       `(when-not ~x\n          (throw (js/Error. ~(core/str \"Assert failed: \" (core/pr-str x)))))))\n  ([x message]\n     (core/when *assert*\n       `(when-not ~x\n          (throw (js/Error.\n                  (cljs.core/str \"Assert failed: \" ~message \"\\n\" ~(core/pr-str x))))))))\n\n(core/defmacro for\n  \"List comprehension. Takes a vector of one or more\n   binding-form/collection-expr pairs, each followed by zero or more\n   modifiers, and yields a lazy sequence of evaluations of expr.\n   Collections are iterated in a nested fashion, rightmost fastest,\n   and nested coll-exprs can refer to bindings created in prior\n   binding-forms.  Supported modifiers are: :let [binding-form expr ...],\n   :while test, :when test.\n\n  (take 100 (for [x (range 100000000) y (range 1000000) :while (< y x)]  [x y]))\"\n  [seq-exprs body-expr]\n  (assert-args for\n    (vector? seq-exprs) \"a vector for its binding\"\n    (even? (count seq-exprs)) \"an even number of forms in binding vector\")\n  (core/let [to-groups (core/fn [seq-exprs]\n                         (reduce (core/fn [groups [k v]]\n                                   (if (core/keyword? k)\n                                     (conj (pop groups) (conj (peek groups) [k v]))\n                                     (conj groups [k v])))\n                           [] (partition 2 seq-exprs)))\n             err (core/fn [& msg] (throw (ex-info (apply core/str msg) {})))\n             emit-bind (core/fn emit-bind [[[bind expr & mod-pairs]\n                                       & [[_ next-expr] :as next-groups]]]\n                         (core/let [giter (gensym \"iter__\")\n                                    gxs (gensym \"s__\")\n                                    do-mod (core/fn do-mod [[[k v :as pair] & etc]]\n                                             (core/cond\n                                               (= k :let) `(let ~v ~(do-mod etc))\n                                               (= k :while) `(when ~v ~(do-mod etc))\n                                               (= k :when) `(if ~v\n                                                              ~(do-mod etc)\n                                                              (recur (rest ~gxs)))\n                                               (core/keyword? k) (err \"Invalid 'for' keyword \" k)\n                                               next-groups\n                                               `(let [iterys# ~(emit-bind next-groups)\n                                                      fs# (seq (iterys# ~next-expr))]\n                                                  (if fs#\n                                                    (concat fs# (~giter (rest ~gxs)))\n                                                    (recur (rest ~gxs))))\n                                               :else `(cons ~body-expr\n                                                        (~giter (rest ~gxs)))))]\n                           (if next-groups\n                             #_ \"not the inner-most loop\"\n                             `(fn ~giter [~gxs]\n                                (lazy-seq\n                                  (loop [~gxs ~gxs]\n                                    (when-first [~bind ~gxs]\n                                      ~(do-mod mod-pairs)))))\n                             #_\"inner-most loop\"\n                             (core/let [gi (gensym \"i__\")\n                                        gb (gensym \"b__\")\n                                        do-cmod (core/fn do-cmod [[[k v :as pair] & etc]]\n                                                  (core/cond\n                                                    (= k :let) `(let ~v ~(do-cmod etc))\n                                                    (= k :while) `(when ~v ~(do-cmod etc))\n                                                    (= k :when) `(if ~v\n                                                                   ~(do-cmod etc)\n                                                                   (recur\n                                                                     (unchecked-inc ~gi)))\n                                                    (core/keyword? k)\n                                                    (err \"Invalid 'for' keyword \" k)\n                                                    :else\n                                                    `(do (chunk-append ~gb ~body-expr)\n                                                         (recur (unchecked-inc ~gi)))))]\n                               `(fn ~giter [~gxs]\n                                  (lazy-seq\n                                    (loop [~gxs ~gxs]\n                                      (when-let [~gxs (seq ~gxs)]\n                                        (if (chunked-seq? ~gxs)\n                                          (let [c# ^not-native (chunk-first ~gxs)\n                                                size# (count c#)\n                                                ~gb (chunk-buffer size#)]\n                                            (if (coercive-boolean\n                                                  (loop [~gi 0]\n                                                    (if (< ~gi size#)\n                                                      (let [~bind (-nth c# ~gi)]\n                                                        ~(do-cmod mod-pairs))\n                                                      true)))\n                                              (chunk-cons\n                                                (chunk ~gb)\n                                                (~giter (chunk-rest ~gxs)))\n                                              (chunk-cons (chunk ~gb) nil)))\n                                          (let [~bind (first ~gxs)]\n                                            ~(do-mod mod-pairs)))))))))))]\n    `(let [iter# ~(emit-bind (to-groups seq-exprs))]\n       (iter# ~(second seq-exprs)))))\n\n(core/defmacro doseq\n  \"Repeatedly executes body (presumably for side-effects) with\n  bindings and filtering as provided by \\\"for\\\".  Does not retain\n  the head of the sequence. Returns nil.\"\n  [seq-exprs & body]\n  (assert-args doseq\n    (vector? seq-exprs) \"a vector for its binding\"\n    (even? (count seq-exprs)) \"an even number of forms in binding vector\")\n  (core/let [err (core/fn [& msg] (throw (ex-info (apply core/str msg) {})))\n             step (core/fn step [recform exprs]\n                    (core/if-not exprs\n                      [true `(do ~@body)]\n                      (core/let [k (first exprs)\n                                 v (second exprs)\n\n                                 seqsym (gensym \"seq__\")\n                                 recform (if (core/keyword? k) recform `(recur (next ~seqsym) nil 0 0))\n                                 steppair (step recform (nnext exprs))\n                                 needrec (steppair 0)\n                                 subform (steppair 1)]\n                        (core/cond\n                          (= k :let) [needrec `(let ~v ~subform)]\n                          (= k :while) [false `(when ~v\n                                                 ~subform\n                                                 ~@(core/when needrec [recform]))]\n                          (= k :when) [false `(if ~v\n                                                (do\n                                                  ~subform\n                                                  ~@(core/when needrec [recform]))\n                                                ~recform)]\n                          (core/keyword? k) (err \"Invalid 'doseq' keyword\" k)\n                          :else (core/let [chunksym (with-meta (gensym \"chunk__\")\n                                                      {:tag 'not-native})\n                                           countsym (gensym \"count__\")\n                                           isym     (gensym \"i__\")\n                                           recform-chunk  `(recur ~seqsym ~chunksym ~countsym (unchecked-inc ~isym))\n                                           steppair-chunk (step recform-chunk (nnext exprs))\n                                           subform-chunk  (steppair-chunk 1)]\n                                  [true `(loop [~seqsym   (seq ~v)\n                                                ~chunksym nil\n                                                ~countsym 0\n                                                ~isym     0]\n                                           (if (coercive-boolean (< ~isym ~countsym))\n                                             (let [~k (-nth ~chunksym ~isym)]\n                                               ~subform-chunk\n                                               ~@(core/when needrec [recform-chunk]))\n                                             (when-let [~seqsym (seq ~seqsym)]\n                                               (if (chunked-seq? ~seqsym)\n                                                 (let [c# (chunk-first ~seqsym)]\n                                                   (recur (chunk-rest ~seqsym) c#\n                                                     (count c#) 0))\n                                                 (let [~k (first ~seqsym)]\n                                                   ~subform\n                                                   ~@(core/when needrec [recform]))))))])))))]\n    (nth (step nil (seq seq-exprs)) 1)))\n\n(core/defmacro array [& rest]\n  (core/let [xs-str (core/->> (repeat \"~{}\")\n                      (take (count rest))\n                      (interpose \",\")\n                      (apply core/str))]\n    (vary-meta\n      (list* 'js* (core/str \"[\" xs-str \"]\") rest)\n      assoc :tag 'array)))\n\n(core/defmacro make-array\n  ([size]\n   (vary-meta\n     (if (core/number? size)\n       `(array ~@(take size (repeat nil)))\n       `(js/Array. ~size))\n     assoc :tag 'array))\n  ([type size]\n   `(cljs.core/make-array ~size))\n  ([type size & more-sizes]\n   (vary-meta\n     `(let [dims#     (list ~@more-sizes)\n            dimarray# (cljs.core/make-array ~size)]\n        (dotimes [i# (alength dimarray#)]\n          (aset dimarray# i# (apply cljs.core/make-array nil dims#)))\n        dimarray#)\n     assoc :tag 'array)))\n\n(core/defmacro list\n  ([]\n   '(.-EMPTY cljs.core/List))\n  ([x & xs]\n   (if (= :constant (:op (cljs.analyzer/analyze &env x)))\n     `(-conj (list ~@xs) ~x)\n     `(let [x# ~x]\n        (-conj (list ~@xs) x#)))))\n\n(core/defmacro vector\n  ([] '(.-EMPTY cljs.core/PersistentVector))\n  ([& xs]\n   (core/let [cnt (count xs)]\n     (if (core/< cnt 32)\n       `(cljs.core/PersistentVector. nil ~cnt 5\n          (.-EMPTY-NODE cljs.core/PersistentVector) (array ~@xs) nil)\n       (vary-meta\n         `(.fromArray cljs.core/PersistentVector (array ~@xs) true)\n         assoc :tag 'cljs.core/PersistentVector)))))\n\n(core/defmacro array-map\n  ([] '(.-EMPTY cljs.core/PersistentArrayMap))\n  ([& kvs]\n   (core/let [keys (map first (partition 2 kvs))]\n     (if (core/and (every? #(= (:op %) :constant)\n                     (map #(cljs.analyzer/analyze &env %) keys))\n           (= (count (into #{} keys)) (count keys)))\n       `(cljs.core/PersistentArrayMap. nil ~(clojure.core// (count kvs) 2) (array ~@kvs) nil)\n       `(.createAsIfByAssoc cljs.core/PersistentArrayMap (array ~@kvs))))))\n\n(core/defmacro hash-map\n  ([] `(.-EMPTY cljs.core/PersistentHashMap))\n  ([& kvs]\n   (core/let [pairs (partition 2 kvs)\n              ks    (map first pairs)\n              vs    (map second pairs)]\n     (vary-meta\n       `(.fromArrays cljs.core/PersistentHashMap (array ~@ks) (array ~@vs))\n       assoc :tag 'cljs.core/PersistentHashMap))))\n\n(core/defmacro hash-set\n  ([] `(.-EMPTY cljs.core/PersistentHashSet))\n  ([& xs]\n    (if (core/and (core/<= (count xs) 8)\n                  (every? #(= (:op %) :constant)\n                    (map #(cljs.analyzer/analyze &env %) xs))\n                  (= (count (into #{} xs)) (count xs)))\n      `(cljs.core/PersistentHashSet. nil\n         (cljs.core/PersistentArrayMap. nil ~(count xs) (array ~@(interleave xs (repeat nil))) nil)\n         nil)\n      (vary-meta\n        `(.createAsIfByAssoc cljs.core/PersistentHashSet (array ~@xs))\n        assoc :tag 'cljs.core/PersistentHashSet))))\n\n(core/defn- js-obj* [kvs]\n  (core/let [kvs-str (core/->> (repeat \"~{}:~{}\")\n                       (take (count kvs))\n                       (interpose \",\")\n                       (apply core/str))]\n    (vary-meta\n      (list* 'js* (core/str \"{\" kvs-str \"}\") (apply concat kvs))\n      assoc :tag 'object)))\n\n(core/defmacro js-obj [& rest]\n  (core/let [sym-or-str? (core/fn [x] (core/or (core/symbol? x) (core/string? x)))\n             filter-on-keys (core/fn [f coll]\n                              (core/->> coll\n                                (filter (core/fn [[k _]] (f k)))\n                                (into {})))\n             kvs (into {} (map vec (partition 2 rest)))\n             sym-pairs (filter-on-keys core/symbol? kvs)\n             expr->local (zipmap\n                           (filter (complement sym-or-str?) (keys kvs))\n                           (repeatedly gensym))\n             obj (gensym \"obj\")]\n    (if (empty? rest)\n      (js-obj* '())\n      `(let [~@(apply concat (clojure.set/map-invert expr->local))\n            ~obj ~(js-obj* (filter-on-keys core/string? kvs))]\n        ~@(map (core/fn [[k v]] `(aset ~obj ~k ~v)) sym-pairs)\n        ~@(map (core/fn [[k v]] `(aset ~obj ~v ~(core/get kvs k))) expr->local)\n        ~obj))))\n\n(core/defmacro alength [a]\n  (vary-meta\n    (core/list 'js* \"~{}.length\" a)\n    assoc :tag 'number))\n\n(core/defmacro amap\n  \"Maps an expression across an array a, using an index named idx, and\n  return value named ret, initialized to a clone of a, then setting\n  each element of ret to the evaluation of expr, returning the new\n  array ret.\"\n  [a idx ret expr]\n  `(let [a# ~a\n         ~ret (cljs.core/aclone a#)]\n     (loop  [~idx 0]\n       (if (< ~idx  (alength a#))\n         (do\n           (aset ~ret ~idx ~expr)\n           (recur (inc ~idx)))\n         ~ret))))\n\n(core/defmacro areduce\n  \"Reduces an expression across an array a, using an index named idx,\n  and return value named ret, initialized to init, setting ret to the\n  evaluation of expr at each step, returning ret.\"\n  [a idx ret init expr]\n  `(let [a# ~a]\n     (loop  [~idx 0 ~ret ~init]\n       (if (< ~idx  (alength a#))\n         (recur (inc ~idx) ~expr)\n         ~ret))))\n\n(core/defmacro dotimes\n  \"bindings => name n\n\n  Repeatedly executes body (presumably for side-effects) with name\n  bound to integers from 0 through n-1.\"\n  [bindings & body]\n  (core/let [i (first bindings)\n             n (second bindings)]\n    `(let [n# ~n]\n       (loop [~i 0]\n         (when (< ~i n#)\n           ~@body\n           (recur (inc ~i)))))))\n\n(core/defn- check-valid-options\n  \"Throws an exception if the given option map contains keys not listed\n  as valid, else returns nil.\"\n  [options & valid-keys]\n  (core/when (seq (apply disj (apply core/hash-set (keys options)) valid-keys))\n    (throw\n      (apply core/str \"Only these options are valid: \"\n        (first valid-keys)\n        (map #(core/str \", \" %) (rest valid-keys))))))\n\n(core/defmacro defmulti\n  \"Creates a new multimethod with the associated dispatch function.\n  The docstring and attribute-map are optional.\n\n  Options are key-value pairs and may be one of:\n    :default    the default dispatch value, defaults to :default\n    :hierarchy  the isa? hierarchy to use for dispatching\n                defaults to the global hierarchy\"\n  [mm-name & options]\n  (core/let [docstring   (if (core/string? (first options))\n                           (first options)\n                           nil)\n             options     (if (core/string? (first options))\n                           (next options)\n                           options)\n             m           (if (map? (first options))\n                           (first options)\n                           {})\n             options     (if (map? (first options))\n                           (next options)\n                           options)\n             dispatch-fn (first options)\n             options     (next options)\n             m           (if docstring\n                           (assoc m :doc docstring)\n                           m)\n             m           (if (meta mm-name)\n                           (conj (meta mm-name) m)\n                           m)\n             mm-ns (core/-> &env :ns :name core/str)]\n    (core/when (= (count options) 1)\n      (throw\n        #?(:clj (Exception. \"The syntax for defmulti has changed. Example: (defmulti name dispatch-fn :default dispatch-value)\")\n           :cljs (js/Error. \"The syntax for defmulti has changed. Example: (defmulti name dispatch-fn :default dispatch-value)\"))))\n    (core/let [options (apply core/hash-map options)\n               default (core/get options :default :default)]\n      (check-valid-options options :default :hierarchy)\n      `(defonce ~(with-meta mm-name m)\n         (let [method-table# (atom {})\n               prefer-table# (atom {})\n               method-cache# (atom {})\n               cached-hierarchy# (atom {})\n               hierarchy# (cljs.core/get ~options :hierarchy (cljs.core/get-global-hierarchy))]\n           (cljs.core/MultiFn. (cljs.core/symbol ~mm-ns ~(name mm-name)) ~dispatch-fn ~default hierarchy#\n             method-table# prefer-table# method-cache# cached-hierarchy#))))))\n\n(core/defmacro defmethod\n  \"Creates and installs a new method of multimethod associated with dispatch-value. \"\n  [multifn dispatch-val & fn-tail]\n  `(-add-method ~(with-meta multifn {:tag 'cljs.core/MultiFn}) ~dispatch-val (fn ~@fn-tail)))\n\n(core/defmacro time\n  \"Evaluates expr and prints the time it took. Returns the value of expr.\"\n  [expr]\n  `(let [start# (system-time)\n         ret# ~expr]\n     (prn (cljs.core/str \"Elapsed time: \"\n            (.toFixed (- (system-time) start#) 6)\n            \" msecs\"))\n     ret#))\n\n(core/defmacro simple-benchmark\n  \"Runs expr iterations times in the context of a let expression with\n  the given bindings, then prints out the bindings and the expr\n  followed by number of iterations and total time. The optional\n  argument print-fn, defaulting to println, sets function used to\n  print the result. expr's string representation will be produced\n  using pr-str in any case.\"\n  [bindings expr iterations & {:keys [print-fn] :or {print-fn 'println}}]\n  (core/let [bs-str   (pr-str bindings)\n             expr-str (pr-str expr)]\n    `(let ~bindings\n       (let [start#   (.getTime (js/Date.))\n             ret#     (dotimes [_# ~iterations] ~expr)\n             end#     (.getTime (js/Date.))\n             elapsed# (- end# start#)]\n         (~print-fn (str ~bs-str \", \" ~expr-str \", \"\n                      ~iterations \" runs, \" elapsed# \" msecs\"))))))\n\n(def ^:private cs (into [] (map (comp gensym core/str core/char) (range 97 118))))\n\n(core/defn- gen-apply-to-helper\n  ([] (gen-apply-to-helper 1))\n  ([n]\n   (if (core/<= n 20)\n     `(let [~(cs (core/dec n)) (-first ~'args)\n            ~'args (-rest ~'args)]\n        (if (== ~'argc ~n)\n          (~'f ~@(take n cs))\n          ~(gen-apply-to-helper (core/inc n))))\n     `(throw (js/Error. \"Only up to 20 arguments supported on functions\")))))\n\n(core/defmacro gen-apply-to []\n  `(do\n     (set! ~'*unchecked-if* true)\n     (defn ~'apply-to [~'f ~'argc ~'args]\n       (let [~'args (seq ~'args)]\n         (if (zero? ~'argc)\n           (~'f)\n           ~(gen-apply-to-helper))))\n     (set! ~'*unchecked-if* false)))\n\n(core/defn- gen-apply-to-simple-helper\n  [f num-args args]\n  (core/let [new-arg-sym (symbol (core/str \"a\" num-args))\n             proto-name (core/str \"cljs$core$IFn$_invoke$arity$\" (core/inc num-args))\n             proto-prop (symbol (core/str \".-\" proto-name))\n             proto-inv (symbol (core/str \".\" proto-name))\n             next-sym (symbol (core/str \"next_\" num-args))\n             all-args (mapv #(symbol (core/str \"a\" %)) (range (core/inc num-args)))]\n    `(let [~new-arg-sym (cljs.core/-first ~args)\n           ~next-sym (cljs.core/next ~args)]\n       (if (nil? ~next-sym)\n         (if (~proto-prop ~f)\n           (~proto-inv ~f ~@all-args)\n           (.call ~f ~f ~@all-args))\n         ~(if (core/<= 19 num-args)\n            ;; We've exhausted all protocols, fallback to .apply:\n            `(let [arr# (cljs.core/array ~@all-args)]\n               (loop [s# ~next-sym]\n                 (when s#\n                   (do (.push arr# (cljs.core/-first s#))\n                       (recur (cljs.core/next s#)))))\n               (.apply ~f ~f arr#))\n            (gen-apply-to-simple-helper f (core/inc num-args) next-sym))))))\n\n(core/defmacro gen-apply-to-simple\n  [f num-args args]\n  (gen-apply-to-simple-helper f num-args args))\n\n(core/defmacro with-out-str\n  \"Evaluates exprs in a context in which *print-fn* is bound to .append\n  on a fresh StringBuffer.  Returns the string created by any nested\n  printing calls.\"\n  [& body]\n  `(let [sb# (js/goog.string.StringBuffer.)]\n     (binding [cljs.core/*print-newline* true\n               cljs.core/*print-fn* (fn [x#] (.append sb# x#))]\n       ~@body)\n     (cljs.core/str sb#)))\n\n(core/defmacro lazy-cat\n  \"Expands to code which yields a lazy sequence of the concatenation\n  of the supplied colls.  Each coll expr is not evaluated until it is\n  needed.\n\n  (lazy-cat xs ys zs) === (concat (lazy-seq xs) (lazy-seq ys) (lazy-seq zs))\"\n  [& colls]\n  `(concat ~@(map #(core/list `lazy-seq %) colls)))\n\n(core/defmacro js-str [s]\n  (core/list 'js* \"''+~{}\" s))\n\n(core/defmacro es6-iterable [ty]\n  `(aset (.-prototype ~ty) cljs.core/ITER_SYMBOL\n     (fn []\n       (this-as this#\n         (cljs.core/es6-iterator this#)))))\n\n(core/defmacro ns-interns\n  \"Returns a map of the intern mappings for the namespace.\"\n  [[quote ns]]\n  (core/assert (core/and (= quote 'quote) (core/symbol? ns))\n    \"Argument to ns-interns must be a quoted symbol\")\n  `(into {}\n     [~@(map\n          (core/fn [[sym _]]\n            `[(symbol ~(name sym)) (var ~(symbol (name ns) (name sym)))])\n          (get-in @env/*compiler* [:cljs.analyzer/namespaces ns :defs]))]))\n\n(core/defmacro ns-unmap\n  \"Removes the mappings for the symbol from the namespace.\"\n  [[quote0 ns] [quote1 sym]]\n  (core/assert (core/and (= quote0 'quote) (core/symbol? ns)\n                         (= quote1 'quote) (core/symbol? sym))\n    \"Arguments to ns-unmap must be quoted symbols\")\n  (swap! env/*compiler* update-in [::ana/namespaces ns :defs] dissoc sym)\n  `(js-delete ~(comp/munge ns) ~(comp/munge (core/str sym))))\n\n(core/defmacro vswap!\n  \"Non-atomically swaps the value of the volatile as if:\n   (apply f current-value-of-vol args). Returns the value that\n   was swapped in.\"\n  [vol f & args]\n  `(-vreset! ~vol (~f (-deref ~vol) ~@args)))\n\n(core/defmacro locking\n  [x & forms]\n  `(do ~@forms))\n\n(core/defmacro require\n  \"Loads libs, skipping any that are already loaded. Each argument is\n  either a libspec that identifies a lib or a flag that modifies how all the identified\n  libs are loaded. Use :require in the ns macro in preference to calling this\n  directly.\n\n  Libs\n\n  A 'lib' is a named set of resources in classpath whose contents define a\n  library of ClojureScript code. Lib names are symbols and each lib is associated\n  with a ClojureScript namespace. A lib's name also locates its root directory\n  within classpath using Java's package name to classpath-relative path mapping.\n  All resources in a lib should be contained in the directory structure under its\n  root directory. All definitions a lib makes should be in its associated namespace.\n\n  'require loads a lib by loading its root resource. The root resource path\n  is derived from the lib name in the following manner:\n  Consider a lib named by the symbol 'x.y.z; it has the root directory\n  <classpath>/x/y/, and its root resource is <classpath>/x/y/z.clj. The root\n  resource should contain code to create the lib's namespace (usually by using\n  the ns macro) and load any additional lib resources.\n\n  Libspecs\n\n  A libspec is a lib name or a vector containing a lib name followed by\n  options expressed as sequential keywords and arguments.\n\n  Recognized options:\n  :as takes a symbol as its argument and makes that symbol an alias to the\n    lib's namespace in the current namespace.\n  :refer takes a list of symbols to refer from the namespace.\n  :refer-macros takes a list of macro symbols to refer from the namespace.\n  :include-macros true causes macros from the namespace to be required.\n  :rename specifies a map from referred var names to different\n    symbols (and can be used to prevent clashes)\n\n\n  Flags\n\n  A flag is a keyword.\n  Recognized flags: :reload, :reload-all, :verbose\n  :reload forces loading of all the identified libs even if they are\n    already loaded\n  :reload-all implies :reload and also forces loading of all libs that the\n    identified libs directly or indirectly load via require or use\n  :verbose triggers printing information about each load, alias, and refer\n\n  Example:\n\n  The following would load the library clojure.string :as string.\n\n  (require '[clojure.string :as string])\"\n  [& args]\n  `(~'ns* ~(cons :require args)))\n\n(core/defmacro require-macros\n  \"Similar to require but only for macros.\"\n  [& args]\n  `(~'ns* ~(cons :require-macros args)))\n\n(core/defmacro use\n  \"Like require, but referring vars specified by the mandatory\n  :only option.\n\n  Example:\n\n  The following would load the library clojure.set while referring\n  the intersection var.\n\n  (use '[clojure.set :only [intersection]])\"\n  [& args]\n  `(~'ns* ~(cons :use args)))\n\n(core/defmacro use-macros\n  \"Similar to use but only for macros.\"\n  [& args]\n  `(~'ns* ~(cons :use-macros args)))\n\n(core/defmacro import\n  \"import-list => (closure-namespace constructor-name-symbols*)\n\n  For each name in constructor-name-symbols, adds a mapping from name to the\n  constructor named by closure-namespace to the current namespace. Use :import in the ns\n  macro in preference to calling this directly.\"\n  [& import-symbols-or-lists]\n  `(~'ns* ~(cons :import import-symbols-or-lists)))\n\n(core/defmacro refer-clojure\n  \"Refers to all the public vars of `cljs.core`, subject to\n  filters.\n  Filters can include at most one each of:\n\n  :exclude list-of-symbols\n  :rename map-of-fromsymbol-tosymbol\n\n  Filters can be used to select a subset, via exclusion, or to provide a mapping\n  to a symbol different from the var's name, in order to prevent clashes.\"\n  [& args]\n  `(~'ns* ~(cons :refer-clojure args)))\n\n;; INTERNAL - do not use, only for Node.js\n(core/defmacro load-file* [f]\n  `(. js/goog (~'nodeGlobalRequire ~f)))\n\n(core/defmacro macroexpand-1\n  \"If form represents a macro form, returns its expansion,\n  else returns form.\"\n  [quoted]\n  (core/assert (core/= (core/first quoted) 'quote)\n    \"Argument to macroexpand-1 must be quoted\")\n  (core/let [form (second quoted)]\n    (if (seq? form)\n      `(quote ~(ana/macroexpand-1 &env form))\n      form)))\n\n(core/defmacro macroexpand\n  \"Repeatedly calls macroexpand-1 on form until it no longer\n  represents a macro form, then returns it.  Note neither\n  macroexpand-1 nor macroexpand expand macros in subforms.\"\n  [quoted]\n  (core/assert (core/= (core/first quoted) 'quote)\n    \"Argument to macroexpand must be quoted\")\n  (core/let [form (second quoted)\n             env &env]\n    (if (seq? form)\n      (core/loop [form form form' (ana/macroexpand-1 env form)]\n        (core/if-not (core/identical? form form')\n          (recur form' (ana/macroexpand-1 env form'))\n          `(quote ~form')))\n      form)))\n\n(core/defn- multi-arity-fn? [fdecl]\n  (core/< 1 (count fdecl)))\n\n(core/defn- variadic-fn? [fdecl]\n  (core/and (= 1 (count fdecl))\n            (some '#{&} (ffirst fdecl))))\n\n(core/defn- variadic-fn*\n  ([sym method]\n   (variadic-fn* sym method true))\n  ([sym [arglist & body :as method] solo]\n   (core/let [sig (remove '#{&} arglist)\n              restarg (gensym \"seq\")]\n     (core/letfn [(get-delegate []\n                    'cljs$core$IFn$_invoke$arity$variadic)\n                  (get-delegate-prop []\n                    (symbol (core/str \"-\" (get-delegate))))\n                  (param-bind [param]\n                    `[~param (^::ana/no-resolve first ~restarg)\n                      ~restarg (^::ana/no-resolve next ~restarg)])\n                  (apply-to []\n                    (if (core/< 1 (count sig))\n                      (core/let [params (repeatedly (core/dec (count sig)) gensym)]\n                        `(fn\n                           ([~restarg]\n                            (let [~@(mapcat param-bind params)]\n                              (. ~sym (~(get-delegate) ~@params ~restarg))))))\n                      `(fn\n                         ([~restarg]\n                          (. ~sym (~(get-delegate) (seq ~restarg)))))))]\n       `(do\n          (set! (. ~sym ~(get-delegate-prop))\n            (fn (~(vec sig) ~@body)))\n          ~@(core/when solo\n              `[(set! (. ~sym ~'-cljs$lang$maxFixedArity)\n                  ~(core/dec (count sig)))])\n          (set! (. ~sym ~'-cljs$lang$applyTo)\n            ~(apply-to)))))))\n\n(core/defmacro copy-arguments [dest]\n  `(let [len# (alength (js-arguments))]\n     (loop [i# 0]\n       (when (< i# len#)\n         (.push ~dest (aget (js-arguments) i#))\n         (recur (inc i#))))))\n\n(core/defn- variadic-fn [name meta [[arglist & body :as method] :as fdecl] emit-var?]\n  (core/letfn [(dest-args [c]\n                 (map (core/fn [n] `(aget (js-arguments) ~n))\n                   (range c)))]\n    (core/let [rname (symbol (core/str ana/*cljs-ns*) (core/str name))\n               sig   (remove '#{&} arglist)\n               c-1   (core/dec (count sig))\n               meta  (assoc meta\n                       :top-fn\n                       {:variadic true\n                        :max-fixed-arity c-1\n                        :method-params [sig]\n                        :arglists (core/list arglist)\n                        :arglists-meta (doall (map meta [arglist]))})]\n      `(do\n         (def ~(with-meta name meta)\n           (fn [~'var_args]\n             (let [args# (array)]\n               (copy-arguments args#)\n               (let [argseq# (when (< ~c-1 (alength args#))\n                               (new ^::ana/no-resolve cljs.core/IndexedSeq\n                                 (.slice args# ~c-1) 0 nil))]\n                 (. ~rname\n                   (~'cljs$core$IFn$_invoke$arity$variadic ~@(dest-args c-1) argseq#))))))\n         ~(variadic-fn* rname method)\n         ~(core/when emit-var? `(var ~name))))))\n\n(core/comment\n  (require '[clojure.pprint :as pp])\n  (pp/pprint (variadic-fn 'foo {} '(([& xs]))))\n  (pp/pprint (variadic-fn 'foo {} '(([a & xs] xs))))\n  (pp/pprint (variadic-fn 'foo {} '(([a b & xs] xs))))\n  (pp/pprint (variadic-fn 'foo {} '(([a [b & cs] & xs] xs))))\n  )\n\n(core/defn- multi-arity-fn [name meta fdecl emit-var?]\n  (core/letfn [(dest-args [c]\n                 (map (core/fn [n] `(aget (js-arguments) ~n))\n                   (range c)))\n               (fixed-arity [rname sig]\n                 (core/let [c (count sig)]\n                   [c `(. ~rname\n                         (~(symbol\n                             (core/str \"cljs$core$IFn$_invoke$arity$\" c))\n                           ~@(dest-args c)))]))\n               (fn-method [[sig & body :as method]]\n                 (if (some '#{&} sig)\n                   (variadic-fn* name method false)\n                   `(set!\n                      (. ~name\n                        ~(symbol (core/str \"-cljs$core$IFn$_invoke$arity$\"\n                                   (count sig))))\n                      (fn ~method))))]\n    (core/let [rname    (symbol (core/str ana/*cljs-ns*) (core/str name))\n               arglists (map first fdecl)\n               varsig?  #(some '#{&} %)\n               variadic (boolean (some varsig? arglists))\n               sigs     (remove varsig? arglists)\n               maxfa    (apply core/max\n                          (concat\n                            (map count sigs)\n                            [(core/- (count (first (filter varsig? arglists))) 2)]))\n               meta     (assoc meta\n                          :top-fn\n                          {:variadic variadic\n                           :max-fixed-arity maxfa\n                           :method-params sigs\n                           :arglists arglists\n                           :arglists-meta (doall (map meta arglists))})\n               args-sym (gensym \"args\")\n               param-counts (map count arglists)]\n      (core/when (not= (distinct param-counts) param-counts)\n        (ana/warning :overload-arity {} {:name name}))\n      `(do\n         (def ~(with-meta name meta)\n           (fn [~'var_args]\n             (case (alength (js-arguments))\n               ~@(mapcat #(fixed-arity rname %) sigs)\n               ~(if variadic\n                  `(let [args-arr# (array)]\n                     (copy-arguments args-arr#)\n                     (let [argseq# (new ^::ana/no-resolve cljs.core/IndexedSeq\n                                        (.slice args-arr# ~maxfa) 0 nil)]\n                       (. ~rname\n                          (~'cljs$core$IFn$_invoke$arity$variadic\n                           ~@(dest-args maxfa)\n                           argseq#))))\n                  (if (:macro meta)\n                    `(throw (js/Error.\n                             (str \"Invalid arity: \" (- (alength (js-arguments)) 2))))\n                    `(throw (js/Error.\n                             (str \"Invalid arity: \" (alength (js-arguments))))))))))\n         ~@(map fn-method fdecl)\n         ;; optimization properties\n         (set! (. ~name ~'-cljs$lang$maxFixedArity) ~maxfa)\n         ~(core/when emit-var? `(var ~name))))))\n\n(core/comment\n  (require '[clojure.pprint :as pp])\n  (pp/pprint (multi-arity-fn 'foo {} '(([a]) ([a b]))))\n  (pp/pprint (multi-arity-fn 'foo {} '(([a]) ([a & xs]))))\n  (pp/pprint (multi-arity-fn 'foo {} '(([a]) ([a [b & cs] & xs]))))\n  ;; CLJS-1216\n  (pp/pprint (multi-arity-fn 'foo {} '(([a]) ([a b & xs]))))\n  )\n\n(def\n  ^{:doc \"Same as (def name (core/fn [params* ] exprs*)) or (def\n    name (core/fn ([params* ] exprs*)+)) with any doc-string or attrs added\n    to the var metadata. prepost-map defines a map with optional keys\n    :pre and :post that contain collections of pre or post conditions.\"\n    :arglists '([name doc-string? attr-map? [params*] prepost-map? body]\n                 [name doc-string? attr-map? ([params*] prepost-map? body)+ attr-map?])}\n  defn (core/fn defn [&form &env name & fdecl]\n         ;; Note: Cannot delegate this check to def because of the call to (with-meta name ..)\n         (if (core/instance? #?(:clj clojure.lang.Symbol :cljs Symbol) name)\n           nil\n           (throw\n             #?(:clj (IllegalArgumentException. \"First argument to defn must be a symbol\")\n                :cljs (js/Error. \"First argument to defn must be a symbol\"))))\n         (core/let [m (if (core/string? (first fdecl))\n                        {:doc (first fdecl)}\n                        {})\n                    fdecl (if (core/string? (first fdecl))\n                            (next fdecl)\n                            fdecl)\n                    m (if (map? (first fdecl))\n                        (conj m (first fdecl))\n                        m)\n                    fdecl (if (map? (first fdecl))\n                            (next fdecl)\n                            fdecl)\n                    fdecl (if (vector? (first fdecl))\n                            (core/list fdecl)\n                            fdecl)\n                    m (if (map? (last fdecl))\n                        (conj m (last fdecl))\n                        m)\n                    fdecl (if (map? (last fdecl))\n                            (butlast fdecl)\n                            fdecl)\n                    m (conj {:arglists (core/list 'quote (sigs fdecl))} m)\n                    ;; no support for :inline\n                    ;m (core/let [inline (:inline m)\n                    ;             ifn (first inline)\n                    ;             iname (second inline)]\n                    ;    ;; same as: (if (and (= 'fn ifn) (not (symbol? iname))) ...)\n                    ;    (if (if #?(:clj (clojure.lang.Util/equiv 'fn ifn)\n                    ;               :cljs (= 'fn ifn))\n                    ;          (if #?(:clj (core/instance? clojure.lang.Symbol iname)\n                    ;                 :cljs (core/instance? Symbol iname)) false true))\n                    ;      ;; inserts the same fn name to the inline fn if it does not have one\n                    ;      (assoc m\n                    ;        :inline (cons ifn\n                    ;                  (cons (clojure.lang.Symbol/intern\n                    ;                          (.concat (.getName ^clojure.lang.Symbol name) \"__inliner\"))\n                    ;                    (next inline))))\n                    ;      m))\n                    m (conj (if (meta name) (meta name) {}) m)]\n           (core/cond\n             (multi-arity-fn? fdecl)\n             (multi-arity-fn name\n               (if (comp/checking-types?)\n                 (update-in m [:jsdoc] conj \"@param {...*} var_args\")\n                 m) fdecl (:def-emits-var &env))\n\n             (variadic-fn? fdecl)\n             (variadic-fn name\n               (if (comp/checking-types?)\n                 (update-in m [:jsdoc] conj \"@param {...*} var_args\")\n                 m) fdecl (:def-emits-var &env))\n\n             :else\n             (core/list 'def (with-meta name m)\n               ;;todo - restore propagation of fn name\n               ;;must figure out how to convey primitive hints to self calls first\n               (cons `fn fdecl))))))\n\n#?(:clj  (. (var defn) (setMacro))\n   :cljs (set! (. defn -cljs$lang$macro) true))\n\n(core/defn defmacro\n  \"Like defn, but the resulting function name is declared as a\n  macro and will be used as a macro by the compiler when it is\n  called.\"\n  {:arglists '([name doc-string? attr-map? [params*] body]\n               [name doc-string? attr-map? ([params*] body)+ attr-map?])\n   :macro true}\n  [&form &env name & args]\n  (core/let [prefix (core/loop [p (core/list (vary-meta name assoc :macro true)) args args]\n                      (core/let [f (first args)]\n                        (if (core/string? f)\n                          (recur (cons f p) (next args))\n                          (if (map? f)\n                            (recur (cons f p) (next args))\n                            p))))\n             fdecl (core/loop [fd args]\n                     (if (core/string? (first fd))\n                       (recur (next fd))\n                       (if (map? (first fd))\n                         (recur (next fd))\n                         fd)))\n             fdecl (if (vector? (first fdecl))\n                     (core/list fdecl)\n                     fdecl)\n             add-implicit-args (core/fn [fd]\n                                 (core/let [args (first fd)]\n                                   (cons (vec (cons '&form (cons '&env args))) (next fd))))\n             add-args (core/fn [acc ds]\n                        (if (core/nil? ds)\n                          acc\n                          (core/let [d (first ds)]\n                            (if (map? d)\n                              (conj acc d)\n                              (recur (conj acc (add-implicit-args d)) (next ds))))))\n             fdecl (seq (add-args [] fdecl))\n             decl (core/loop [p prefix d fdecl]\n                    (if p\n                      (recur (next p) (cons (first p) d))\n                      d))]\n    `(let [ret# ~(cons `defn decl)]\n       (set! (. ~name ~'-cljs$lang$macro) true)\n       ret#)))\n\n#?(:clj  (. (var defmacro) (setMacro))\n   :cljs (set! (. defmacro -cljs$lang$macro) true))\n\n(core/defmacro resolve\n  \"Returns the var to which a symbol will be resolved in the namespace else nil.\"\n  [[_ sym]]\n  (core/let [env &env\n             [var meta] (try\n                          (core/let [var (ana/resolve-var env sym (ana/confirm-var-exists-throw)) ]\n                            [var (ana/var-meta var)])\n                          (catch #?@(:clj [Throwable t] :cljs [:default e])\n                            [(ana/resolve-var env sym) nil]))\n             resolved (vary-meta (:name var) assoc ::ana/no-resolve true)]\n    `(when (exists? ~resolved)\n       (cljs.core/Var. (fn [] ~resolved) '~resolved ~meta))))"))
(comment
  (let [s "\n;a\n;b\n\n;c\nd\ne\n;e\n"]
    (prn s)
    (prn (emit/string (ast s)))
    (= s (emit/string (ast s)))))

